<?php
/**
 * API de Moderação de Comentários
 * Endpoint principal para moderação usando OpenAI API
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/openai.php';
require_once __DIR__ . '/../config/thresholds.php';

try {
    error_log("moderate.php - Início da requisição");
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método não permitido');
    }

    $input = file_get_contents('php://input');
    error_log("moderate.php - Input recebido: " . substr($input, 0, 200));
    
    $data = json_decode($input, true);

    if (!$data) {
        $data = $_POST;
    }
    
    error_log("moderate.php - Data decoded: " . print_r($data, true));

    // Validar input
    if (empty($data['input'])) {
        throw new Exception('Campo "input" é obrigatório');
    }

    $inputText = $data['input'];
    $isMultiple = is_array($inputText);
    
    // Verificar se deve pular salvamento no banco (modo rápido)
    $skipSave = isset($data['skip_save']) && $data['skip_save'] === true;
    
    error_log("moderate.php - Input text count: " . ($isMultiple ? count($inputText) : 1));
    error_log("moderate.php - Skip save: " . ($skipSave ? 'true' : 'false'));

    // Inicializar OpenAI
    error_log("moderate.php - Inicializando OpenAI");
    $openai = new OpenAIModeration();
    
    // Chamar API de moderação
    error_log("moderate.php - Chamando API OpenAI");
    $response = $openai->moderate($inputText);
    error_log("moderate.php - Resposta OpenAI recebida");
    
    // Aplicar thresholds customizados
    error_log("moderate.php - Aplicando thresholds");
    $thresholds = new ModerationThresholds();
    $response = $thresholds->applyThresholds($response);
    error_log("moderate.php - Thresholds aplicados");

    // Processar resultados
    $results = [];
    
    if ($skipSave) {
        // Modo rápido - não salvar no banco
        error_log("moderate.php - Modo rápido (sem salvar no banco)");
        
        if ($isMultiple) {
            foreach ($inputText as $index => $text) {
                $results[] = $openai->processResult($response, $index);
            }
        } else {
            $results = $openai->processResult($response, 0);
        }
    } else {
        // Modo completo - salvar no banco
        error_log("moderate.php - Modo completo (salvando no banco)");
        
        // Conectar ao banco
        $db = Database::getInstance();
        $conn = $db->getConnection();
        
        if ($isMultiple) {
            // Múltiplos inputs
            foreach ($inputText as $index => $text) {
                $result = processModerationResult($db, $openai, $response, $text, $index);
                $results[] = $result;
            }
        } else {
            // Input único
            $result = processModerationResult($db, $openai, $response, $inputText, 0);
            $results = $result;
        }
    }

    echo json_encode([
        'success' => true,
        'results' => $results,
        'model' => OPENAI_MODEL
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

} catch (Exception $e) {
    $errorMessage = $e->getMessage();
    $httpCode = 400;
    
    error_log("moderate.php - ERRO: " . $errorMessage);
    error_log("moderate.php - Stack trace: " . $e->getTraceAsString());
    
    // Se for erro 429, retornar código apropriado
    if (strpos($errorMessage, '429') !== false) {
        $httpCode = 429;
    }
    
    http_response_code($httpCode);
    echo json_encode([
        'success' => false,
        'error' => $errorMessage,
        'error_code' => $httpCode,
        'error_file' => basename($e->getFile()),
        'error_line' => $e->getLine()
    ], JSON_UNESCAPED_UNICODE);
}

/**
 * Processa resultado de moderação e salva no banco
 */
function processModerationResult($db, $openai, $apiResponse, $text, $index) {
    $conn = $db->getConnection();
    
    // Processar resultado (apiResponse já tem thresholds aplicados)
    $processed = $openai->processResult($apiResponse, $index);
    
    // Debug: verificar valores processados
    // error_log("Processed flagged: " . ($processed['flagged'] ? 'true' : 'false'));
    // error_log("Processed categories: " . json_encode($processed['categories']));
    
    // Salvar comentário na tabela comments_ia (tabela específica para IA)
    $stmt = $conn->prepare("INSERT INTO comments_ia (content, flagged, moderation_id) VALUES (?, ?, ?)");
    $moderationId = $apiResponse['id'] ?? null;
    $stmt->bind_param("sis", $text, $processed['flagged'], $moderationId);
    $stmt->execute();
    $commentId = $db->getLastInsertId();
    $stmt->close();

    // Salvar categorias
    $categories = $processed['categories'] ?? [];
    $scores = $processed['category_scores'] ?? [];
    
    // Garantir que todas as categorias conhecidas sejam salvas
    $allCategories = OpenAIModeration::getCategories();
    
    $stmt = $conn->prepare("INSERT INTO moderation_categories (comment_id, category_name, category_detected, category_score) VALUES (?, ?, ?, ?)");
    
    foreach ($allCategories as $categoryName => $categoryLabel) {
        // Obter valor detectado (padrão: false se não existir)
        $detected = isset($categories[$categoryName]) ? $categories[$categoryName] : false;
        $score = isset($scores[$categoryName]) ? floatval($scores[$categoryName]) : 0.0;
        
        // Garantir que $detected seja um boolean válido convertido para inteiro (0 ou 1)
        $detectedInt = ($detected === true || $detected === 1 || $detected === '1' || $detected === 'true') ? 1 : 0;
        
        $stmt->bind_param("isid", $commentId, $categoryName, $detectedInt, $score);
        $stmt->execute();
    }
    $stmt->close();

    // Salvar histórico completo
    $stmt = $conn->prepare("INSERT INTO moderation_history (comment_id, moderation_response, api_model, api_id) VALUES (?, ?, ?, ?)");
    $responseJson = json_encode($apiResponse, JSON_UNESCAPED_UNICODE);
    $model = OPENAI_MODEL;
    $apiId = $apiResponse['id'] ?? null;
    $stmt->bind_param("isss", $commentId, $responseJson, $model, $apiId);
    $stmt->execute();
    $stmt->close();

    // Retornar resultado formatado
    return [
        'comment_id' => $commentId,
        'content' => $text,
        'flagged' => $processed['flagged'],
        'categories' => $processed['categories'],
        'category_scores' => $processed['category_scores'],
        'moderation_id' => $moderationId
    ];
}

