<?php
/**
 * API de Estatísticas
 * Endpoint para obter estatísticas e relatórios de moderação
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../config/database.php';

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        throw new Exception('Método não permitido');
    }

    $db = Database::getInstance();
    $conn = $db->getConnection();

    // Parâmetros opcionais
    $startDate = $_GET['start_date'] ?? null;
    $endDate = $_GET['end_date'] ?? null;
    $type = $_GET['type'] ?? 'overview'; // overview, categories, accuracy, timeline

    $stats = [];

    switch ($type) {
        case 'overview':
            $stats = getOverviewStats($conn, $startDate, $endDate);
            break;
        case 'categories':
            $stats = getCategoryStats($conn, $startDate, $endDate);
            break;
        case 'accuracy':
            $stats = getAccuracyStats($conn, $startDate, $endDate);
            break;
        case 'timeline':
            $stats = getTimelineStats($conn, $startDate, $endDate);
            break;
        default:
            throw new Exception('Tipo de estatística inválido');
    }

    echo json_encode([
        'success' => true,
        'type' => $type,
        'data' => $stats,
        'date_range' => [
            'start' => $startDate,
            'end' => $endDate
        ]
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

/**
 * Estatísticas gerais
 */
function getOverviewStats($conn, $startDate, $endDate) {
    $where = "WHERE 1=1";
    if ($startDate) {
        $where .= " AND DATE(created_at) >= '" . $conn->real_escape_string($startDate) . "'";
    }
    if ($endDate) {
        $where .= " AND DATE(created_at) <= '" . $conn->real_escape_string($endDate) . "'";
    }

    // Total de comentários
    $result = $conn->query("SELECT COUNT(*) as total FROM comments $where");
    $total = $result->fetch_assoc()['total'];

    // Comentários sinalizados
    $result = $conn->query("SELECT COUNT(*) as flagged FROM comments $where AND flagged = 1");
    $flagged = $result->fetch_assoc()['flagged'];

    // Taxa de sinalização
    $flagRate = $total > 0 ? round(($flagged / $total) * 100, 2) : 0;

    // Total de feedbacks
    $result = $conn->query("SELECT COUNT(*) as total_feedback FROM feedback f 
                            INNER JOIN comments c ON f.comment_id = c.id $where");
    $totalFeedback = $result->fetch_assoc()['total_feedback'];

    // Falsos positivos/negativos
    $result = $conn->query("SELECT COUNT(*) as false_count FROM feedback 
                            WHERE is_correct = 0");
    $falseCount = $result->fetch_assoc()['false_count'];

    return [
        'total_comments' => intval($total),
        'flagged_comments' => intval($flagged),
        'approved_comments' => intval($total - $flagged),
        'flag_rate' => $flagRate,
        'total_feedback' => intval($totalFeedback),
        'false_positives_negatives' => intval($falseCount),
        'accuracy_rate' => $totalFeedback > 0 ? round((($totalFeedback - $falseCount) / $totalFeedback) * 100, 2) : 0
    ];
}

/**
 * Estatísticas por categoria
 */
function getCategoryStats($conn, $startDate, $endDate) {
    $where = "WHERE 1=1";
    if ($startDate) {
        $where .= " AND DATE(c.created_at) >= '" . $conn->real_escape_string($startDate) . "'";
    }
    if ($endDate) {
        $where .= " AND DATE(c.created_at) <= '" . $conn->real_escape_string($endDate) . "'";
    }

    $result = $conn->query("
        SELECT 
            mc.category_name,
            COUNT(*) as detected_count,
            AVG(mc.category_score) as avg_score,
            MAX(mc.category_score) as max_score
        FROM moderation_categories mc
        INNER JOIN comments c ON mc.comment_id = c.id
        $where
        AND mc.category_detected = 1
        GROUP BY mc.category_name
        ORDER BY detected_count DESC
    ");

    $categories = [];
    while ($row = $result->fetch_assoc()) {
        $categories[] = [
            'category' => $row['category_name'],
            'detected_count' => intval($row['detected_count']),
            'avg_score' => round(floatval($row['avg_score']), 4),
            'max_score' => round(floatval($row['max_score']), 4)
        ];
    }

    return $categories;
}

/**
 * Estatísticas de precisão
 */
function getAccuracyStats($conn, $startDate, $endDate) {
    $where = "WHERE 1=1";
    if ($startDate) {
        $where .= " AND DATE(c.created_at) >= '" . $conn->real_escape_string($startDate) . "'";
    }
    if ($endDate) {
        $where .= " AND DATE(c.created_at) <= '" . $conn->real_escape_string($endDate) . "'";
    }

    // Feedback correto vs incorreto
    $result = $conn->query("
        SELECT 
            f.is_correct,
            COUNT(*) as count
        FROM feedback f
        INNER JOIN comments c ON f.comment_id = c.id
        $where
        GROUP BY f.is_correct
    ");

    $correct = 0;
    $incorrect = 0;

    while ($row = $result->fetch_assoc()) {
        if ($row['is_correct']) {
            $correct = intval($row['count']);
        } else {
            $incorrect = intval($row['count']);
        }
    }

    $total = $correct + $incorrect;
    $accuracy = $total > 0 ? round(($correct / $total) * 100, 2) : 0;

    // Falsos positivos vs falsos negativos
    $result = $conn->query("
        SELECT 
            f.user_judgment,
            c.flagged,
            COUNT(*) as count
        FROM feedback f
        INNER JOIN comments c ON f.comment_id = c.id
        $where
        AND f.is_correct = 0
        GROUP BY f.user_judgment, c.flagged
    ");

    $falsePositives = 0;
    $falseNegatives = 0;

    while ($row = $result->fetch_assoc()) {
        if ($row['user_judgment'] === 'good' && $row['flagged']) {
            $falsePositives += intval($row['count']);
        } elseif ($row['user_judgment'] === 'bad' && !$row['flagged']) {
            $falseNegatives += intval($row['count']);
        }
    }

    return [
        'total_feedback' => $total,
        'correct' => $correct,
        'incorrect' => $incorrect,
        'accuracy_rate' => $accuracy,
        'false_positives' => $falsePositives,
        'false_negatives' => $falseNegatives
    ];
}

/**
 * Estatísticas de timeline
 */
function getTimelineStats($conn, $startDate, $endDate) {
    $where = "WHERE 1=1";
    if ($startDate) {
        $where .= " AND DATE(created_at) >= '" . $conn->real_escape_string($startDate) . "'";
    }
    if ($endDate) {
        $where .= " AND DATE(created_at) <= '" . $conn->real_escape_string($endDate) . "'";
    }

    $result = $conn->query("
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as total,
            SUM(flagged) as flagged
        FROM comments
        $where
        GROUP BY DATE(created_at)
        ORDER BY date ASC
    ");

    $timeline = [];
    while ($row = $result->fetch_assoc()) {
        $timeline[] = [
            'date' => $row['date'],
            'total' => intval($row['total']),
            'flagged' => intval($row['flagged']),
            'approved' => intval($row['total'] - $row['flagged'])
        ];
    }

    return $timeline;
}

