/**
 * JavaScript para Gerenciar Thresholds
 */

const CATEGORY_NAMES = {
    'hate': 'Ódio',
    'hate/threatening': 'Ódio/Ameaçador',
    'harassment': 'Assédio',
    'harassment/threatening': 'Assédio/Ameaçador',
    'self-harm': 'Autolesão',
    'self-harm/intent': 'Autolesão/Intenção',
    'self-harm/instructions': 'Autolesão/Instruções',
    'sexual': 'Sexual',
    'sexual/minors': 'Sexual/Menores',
    'violence': 'Violência',
    'violence/graphic': 'Violência/Gráfico',
    'illicit': 'Ilícito',
    'illicit/violent': 'Ilícito/Violento'
};

document.addEventListener('DOMContentLoaded', function() {
    loadThresholds();
});

/**
 * Carrega thresholds do servidor
 */
async function loadThresholds() {
    const thresholdsList = document.getElementById('thresholdsList');
    
    try {
        const response = await fetch('api/thresholds.php');
        const data = await response.json();

        if (data.success) {
            displayThresholds(data.thresholds);
        } else {
            throw new Error(data.error || 'Erro ao carregar thresholds');
        }
    } catch (error) {
        console.error('Erro:', error);
        thresholdsList.innerHTML = `
            <div class="alert alert-danger">
                <i class="bi bi-exclamation-triangle"></i> Erro ao carregar thresholds: ${error.message}
            </div>
        `;
    }
}

/**
 * Exibe thresholds na interface
 */
function displayThresholds(thresholds) {
    const thresholdsList = document.getElementById('thresholdsList');
    
    if (thresholds.length === 0) {
        thresholdsList.innerHTML = '<div class="alert alert-info">Nenhum threshold configurado.</div>';
        return;
    }

    let html = '<div class="table-responsive"><table class="table table-hover">';
    html += '<thead><tr><th>Categoria</th><th>Threshold</th><th>Status</th><th>Ações</th></tr></thead><tbody>';

    thresholds.forEach(threshold => {
        const categoryName = CATEGORY_NAMES[threshold.category] || threshold.category;
        const thresholdValue = (threshold.threshold * 100).toFixed(1);
        const isEnabled = threshold.enabled;
        
        html += `
            <tr id="row-${threshold.category}">
                <td>
                    <strong>${categoryName}</strong>
                    <br><small class="text-muted">${threshold.category}</small>
                </td>
                <td>
                    <div class="input-group" style="max-width: 200px;">
                        <input 
                            type="number" 
                            class="form-control threshold-input" 
                            id="threshold-${threshold.category}"
                            value="${threshold.threshold}" 
                            min="0" 
                            max="1" 
                            step="0.01"
                            data-category="${threshold.category}"
                        >
                        <span class="input-group-text">(0-1)</span>
                    </div>
                    <small class="text-muted">Equivale a ${thresholdValue}% de confiança</small>
                </td>
                <td>
                    <div class="form-check form-switch">
                        <input 
                            class="form-check-input enabled-switch" 
                            type="checkbox" 
                            id="enabled-${threshold.category}"
                            ${isEnabled ? 'checked' : ''}
                            data-category="${threshold.category}"
                        >
                        <label class="form-check-label" for="enabled-${threshold.category}">
                            ${isEnabled ? '<span class="badge bg-success">Ativo</span>' : '<span class="badge bg-secondary">Inativo</span>'}
                        </label>
                    </div>
                </td>
                <td>
                    <button 
                        class="btn btn-sm btn-primary" 
                        onclick="saveThreshold('${threshold.category}')"
                    >
                        <i class="bi bi-save"></i> Salvar
                    </button>
                </td>
            </tr>
        `;
    });

    html += '</tbody></table></div>';
    
    // Botão para salvar todos
    html += `
        <div class="mt-3">
            <button class="btn btn-success" onclick="saveAllThresholds()">
                <i class="bi bi-save"></i> Salvar Todas as Alterações
            </button>
            <button class="btn btn-secondary ms-2" onclick="resetToDefaults()">
                <i class="bi bi-arrow-counterclockwise"></i> Restaurar Padrões
            </button>
        </div>
    `;

    thresholdsList.innerHTML = html;

    // Adicionar event listeners para mudanças
    document.querySelectorAll('.threshold-input').forEach(input => {
        input.addEventListener('change', function() {
            const value = parseFloat(this.value);
            if (value < 0 || value > 1) {
                alert('O threshold deve estar entre 0.0 e 1.0');
                this.value = Math.max(0, Math.min(1, value));
            }
        });
    });
}

/**
 * Salva um threshold específico
 */
async function saveThreshold(category) {
    const thresholdInput = document.getElementById(`threshold-${category}`);
    const enabledSwitch = document.getElementById(`enabled-${category}`);
    
    const threshold = parseFloat(thresholdInput.value);
    const enabled = enabledSwitch.checked;

    if (isNaN(threshold) || threshold < 0 || threshold > 1) {
        alert('O threshold deve estar entre 0.0 e 1.0');
        return;
    }

    try {
        const response = await fetch('api/thresholds.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                category: category,
                threshold: threshold,
                enabled: enabled
            })
        });

        const data = await response.json();

        if (data.success) {
            // Feedback visual
            const row = document.getElementById(`row-${category}`);
            row.classList.add('table-success');
            setTimeout(() => {
                row.classList.remove('table-success');
            }, 2000);
            
            // Atualizar valor exibido
            const thresholdValue = (threshold * 100).toFixed(1);
            thresholdInput.parentElement.nextElementSibling.textContent = `Equivale a ${thresholdValue}% de confiança`;
        } else {
            throw new Error(data.error || 'Erro ao salvar threshold');
        }
    } catch (error) {
        console.error('Erro:', error);
        alert('Erro ao salvar threshold: ' + error.message);
    }
}

/**
 * Salva todos os thresholds
 */
async function saveAllThresholds() {
    const thresholds = document.querySelectorAll('.threshold-input');
    let saved = 0;
    let errors = 0;

    for (const input of thresholds) {
        const category = input.dataset.category;
        try {
            await saveThreshold(category);
            saved++;
        } catch (error) {
            errors++;
        }
    }

    if (errors === 0) {
        alert(`Todos os ${saved} thresholds foram salvos com sucesso!`);
    } else {
        alert(`Salvos: ${saved}, Erros: ${errors}`);
    }
}

/**
 * Restaura thresholds para valores padrão (0.5)
 */
async function resetToDefaults() {
    if (!confirm('Tem certeza que deseja restaurar todos os thresholds para os valores padrão (0.5)?')) {
        return;
    }

    const thresholds = document.querySelectorAll('.threshold-input');
    let saved = 0;

    for (const input of thresholds) {
        const category = input.dataset.category;
        input.value = '0.5';
        
        try {
            await saveThreshold(category);
            saved++;
        } catch (error) {
            console.error('Erro ao restaurar:', category, error);
        }
    }

    alert(`${saved} thresholds restaurados para valores padrão.`);
    loadThresholds();
}

