<?php
/**
 * API de Feedback de Treinamento
 * Endpoint para salvar feedback sobre moderações
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../config/database.php';

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Método não permitido');
    }

    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (!$data) {
        $data = $_POST;
    }

    // Validar campos obrigatórios
    if (empty($data['comment_id'])) {
        throw new Exception('Campo "comment_id" é obrigatório');
    }

    if (!isset($data['user_judgment'])) {
        throw new Exception('Campo "user_judgment" é obrigatório (deve ser "good" ou "bad")');
    }

    if (!in_array($data['user_judgment'], ['good', 'bad'])) {
        throw new Exception('Campo "user_judgment" deve ser "good" ou "bad"');
    }

    $commentId = intval($data['comment_id']);
    $userJudgment = $data['user_judgment'];
    $notes = $data['notes'] ?? null;

    // Conectar ao banco
    $db = Database::getInstance();
    $conn = $db->getConnection();

    // Verificar se comentário existe e obter flagged
    $stmt = $conn->prepare("SELECT flagged FROM comments WHERE id = ?");
    $stmt->bind_param("i", $commentId);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        throw new Exception('Comentário não encontrado');
    }

    $comment = $result->fetch_assoc();
    $flagged = $comment['flagged'];
    $stmt->close();

    // Determinar se a moderação estava correta
    // Se o usuário marcou como "bad" e foi flagged = correto
    // Se o usuário marcou como "good" e não foi flagged = correto
    $isCorrect = false;
    if (($userJudgment === 'bad' && $flagged) || ($userJudgment === 'good' && !$flagged)) {
        $isCorrect = true;
    }

    // Salvar feedback
    $stmt = $conn->prepare("INSERT INTO feedback (comment_id, is_correct, user_judgment, notes) VALUES (?, ?, ?, ?)");
    $stmt->bind_param("iiss", $commentId, $isCorrect, $userJudgment, $notes);
    $stmt->execute();
    $feedbackId = $db->getLastInsertId();
    $stmt->close();

    // Atualizar estatísticas
    updateStats($db, $isCorrect);

    echo json_encode([
        'success' => true,
        'feedback_id' => $feedbackId,
        'is_correct' => $isCorrect,
        'message' => $isCorrect ? 'Moderação estava correta' : 'Moderação estava incorreta (falso ' . ($flagged ? 'positivo' : 'negativo') . ')'
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

/**
 * Atualiza estatísticas de moderação
 */
function updateStats($db, $isCorrect) {
    $conn = $db->getConnection();
    $today = date('Y-m-d');

    // Verificar se já existe estatística para hoje
    $stmt = $conn->prepare("SELECT id FROM moderation_stats WHERE stat_date = ?");
    $stmt->bind_param("s", $today);
    $stmt->execute();
    $result = $stmt->get_result();
    $stmt->close();

    if ($result->num_rows > 0) {
        // Atualizar existente
        if (!$isCorrect) {
            // Incrementar falsos positivos ou negativos
            // Isso seria melhorado com mais contexto, mas por simplicidade:
            $stmt = $conn->prepare("UPDATE moderation_stats SET false_positives = false_positives + 1 WHERE stat_date = ?");
            $stmt->bind_param("s", $today);
            $stmt->execute();
            $stmt->close();
        }
    } else {
        // Criar nova entrada
        $falsePositives = $isCorrect ? 0 : 1;
        $stmt = $conn->prepare("INSERT INTO moderation_stats (stat_date, false_positives) VALUES (?, ?)");
        $stmt->bind_param("si", $today, $falsePositives);
        $stmt->execute();
        $stmt->close();
    }
}

