/**
 * JavaScript do Dashboard
 */

let charts = {};

document.addEventListener('DOMContentLoaded', function() {
    // Configurar datas padrão (últimos 30 dias)
    const endDate = new Date();
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - 30);

    document.getElementById('endDate').value = endDate.toISOString().split('T')[0];
    document.getElementById('startDate').value = startDate.toISOString().split('T')[0];

    // Carregar dados
    loadDashboardData();

    // Event listeners para filtros
    document.getElementById('startDate').addEventListener('change', loadDashboardData);
    document.getElementById('endDate').addEventListener('change', loadDashboardData);
});

/**
 * Carrega todos os dados do dashboard
 */
async function loadDashboardData() {
    const startDate = document.getElementById('startDate').value;
    const endDate = document.getElementById('endDate').value;

    const params = new URLSearchParams();
    if (startDate) params.append('start_date', startDate);
    if (endDate) params.append('end_date', endDate);

    try {
        // Carregar estatísticas gerais
        const overview = await apiRequest(`stats.php?type=overview&${params}`);
        if (overview.success) {
            displayOverviewStats(overview.data);
        }

        // Carregar estatísticas de categorias
        const categories = await apiRequest(`stats.php?type=categories&${params}`);
        if (categories.success) {
            displayCategoryChart(categories.data);
        }

        // Carregar estatísticas de precisão
        const accuracy = await apiRequest(`stats.php?type=accuracy&${params}`);
        if (accuracy.success) {
            displayAccuracyCharts(accuracy.data);
        }

        // Carregar timeline
        const timeline = await apiRequest(`stats.php?type=timeline&${params}`);
        if (timeline.success) {
            displayTimelineChart(timeline.data);
        }

    } catch (error) {
        console.error('Erro ao carregar dados:', error);
        alert('Erro ao carregar dados do dashboard');
    }
}

/**
 * Exibe estatísticas gerais em cards
 */
function displayOverviewStats(data) {
    const cardsDiv = document.getElementById('statsCards');
    
    cardsDiv.innerHTML = `
        <div class="col-md-3">
            <div class="stat-card bg-primary text-white">
                <div class="stat-label">Total de Comentários</div>
                <div class="stat-value">${data.total_comments}</div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card bg-danger text-white">
                <div class="stat-label">Sinalizados</div>
                <div class="stat-value">${data.flagged_comments}</div>
                <div class="stat-label">${data.flag_rate}%</div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card bg-success text-white">
                <div class="stat-label">Aprovados</div>
                <div class="stat-value">${data.approved_comments}</div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card bg-info text-white">
                <div class="stat-label">Taxa de Precisão</div>
                <div class="stat-value">${data.accuracy_rate}%</div>
                <div class="stat-label">${data.total_feedback} feedbacks</div>
            </div>
        </div>
    `;
}

/**
 * Exibe gráfico de distribuição
 */
function displayDistributionChart(approved, flagged) {
    const ctx = document.getElementById('distributionChart');
    
    if (charts.distribution) {
        charts.distribution.destroy();
    }

    charts.distribution = new Chart(ctx, {
        type: 'doughnut',
        data: {
            labels: ['Aprovados', 'Sinalizados'],
            datasets: [{
                data: [approved, flagged],
                backgroundColor: ['#198754', '#dc3545'],
                borderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
}

/**
 * Exibe gráfico de timeline
 */
function displayTimelineChart(data) {
    const ctx = document.getElementById('timelineChart');
    
    if (charts.timeline) {
        charts.timeline.destroy();
    }

    const labels = data.map(item => item.date);
    const totalData = data.map(item => item.total);
    const flaggedData = data.map(item => item.flagged);
    const approvedData = data.map(item => item.approved);

    charts.timeline = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [
                {
                    label: 'Total',
                    data: totalData,
                    borderColor: '#0d6efd',
                    backgroundColor: 'rgba(13, 110, 253, 0.1)',
                    tension: 0.4
                },
                {
                    label: 'Sinalizados',
                    data: flaggedData,
                    borderColor: '#dc3545',
                    backgroundColor: 'rgba(220, 53, 69, 0.1)',
                    tension: 0.4
                },
                {
                    label: 'Aprovados',
                    data: approvedData,
                    borderColor: '#198754',
                    backgroundColor: 'rgba(25, 135, 84, 0.1)',
                    tension: 0.4
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true
                }
            },
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });

    // Atualizar gráfico de distribuição com dados totais
    const totalApproved = approvedData.reduce((a, b) => a + b, 0);
    const totalFlagged = flaggedData.reduce((a, b) => a + b, 0);
    displayDistributionChart(totalApproved, totalFlagged);
}

/**
 * Exibe gráfico de categorias
 */
function displayCategoryChart(data) {
    const ctx = document.getElementById('categoryChart');
    
    if (charts.category) {
        charts.category.destroy();
    }

    const labels = data.map(item => formatCategory(item.category));
    const counts = data.map(item => item.detected_count);
    const avgScores = data.map(item => item.avg_score * 100);

    charts.category = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [
                {
                    label: 'Detecções',
                    data: counts,
                    backgroundColor: '#ffc107',
                    yAxisID: 'y'
                },
                {
                    label: 'Pontuação Média (%)',
                    data: avgScores,
                    type: 'line',
                    borderColor: '#0dcaf0',
                    backgroundColor: 'rgba(13, 202, 240, 0.1)',
                    yAxisID: 'y1'
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    type: 'linear',
                    display: true,
                    position: 'left',
                    beginAtZero: true
                },
                y1: {
                    type: 'linear',
                    display: true,
                    position: 'right',
                    beginAtZero: true,
                    max: 100,
                    grid: {
                        drawOnChartArea: false
                    }
                }
            },
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
}

/**
 * Exibe gráficos de precisão
 */
function displayAccuracyCharts(data) {
    // Gráfico de precisão geral
    const ctxAccuracy = document.getElementById('accuracyChart');
    
    if (charts.accuracy) {
        charts.accuracy.destroy();
    }

    charts.accuracy = new Chart(ctxAccuracy, {
        type: 'doughnut',
        data: {
            labels: ['Corretos', 'Incorretos'],
            datasets: [{
                data: [data.correct, data.incorrect],
                backgroundColor: ['#198754', '#dc3545'],
                borderWidth: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom'
                },
                title: {
                    display: true,
                    text: `Taxa de Precisão: ${data.accuracy_rate}%`
                }
            }
        }
    });

    // Gráfico de falsos positivos/negativos
    const ctxFalse = document.getElementById('falseChart');
    
    if (charts.false) {
        charts.false.destroy();
    }

    charts.false = new Chart(ctxFalse, {
        type: 'bar',
        data: {
            labels: ['Falsos Positivos', 'Falsos Negativos'],
            datasets: [{
                label: 'Quantidade',
                data: [data.false_positives, data.false_negatives],
                backgroundColor: ['#ffc107', '#dc3545']
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true
                }
            },
            plugins: {
                legend: {
                    display: false
                }
            }
        }
    });
}

