<?php
/**
 * Sistema de Thresholds Customizados
 * Permite ajustar a sensibilidade da moderação por categoria
 */

require_once __DIR__ . '/database.php';

class ModerationThresholds {
    private $db;
    private $thresholds = null;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Carrega thresholds do banco de dados
     * @return array Array de thresholds [category => threshold_value]
     */
    public function loadThresholds() {
        if ($this->thresholds !== null) {
            return $this->thresholds;
        }

        $conn = $this->db->getConnection();
        $result = $conn->query("
            SELECT category_name, threshold_value, enabled 
            FROM moderation_thresholds 
            WHERE enabled = 1
        ");

        $this->thresholds = [];
        while ($row = $result->fetch_assoc()) {
            $this->thresholds[$row['category_name']] = floatval($row['threshold_value']);
        }

        return $this->thresholds;
    }

    /**
     * Aplica thresholds customizados aos resultados da API
     * @param array $apiResult Resultado original da API
     * @return array Resultado com thresholds aplicados
     */
    public function applyThresholds($apiResult) {
        // Limpar cache para garantir que carregue os thresholds mais recentes
        $this->thresholds = null;
        $thresholds = $this->loadThresholds();
        
        // Se não houver thresholds configurados, usar resultado original da API
        if (empty($thresholds)) {
            return $apiResult;
        }

        // Processar cada resultado
        foreach ($apiResult['results'] as $index => &$result) {
            $categories = $result['categories'] ?? [];
            $scores = $result['category_scores'] ?? [];
            $flagged = false;

            // Aplicar thresholds customizados
            // Processar TODAS as categorias que a API retornou
            foreach ($scores as $category => $score) {
                $score = floatval($score);
                
                // Determinar qual threshold usar
                if (isset($thresholds[$category])) {
                    // Usar threshold customizado do banco
                    $threshold = floatval($thresholds[$category]);
                    
                    // Aplicar threshold customizado
                    if ($score >= $threshold) {
                        $categories[$category] = true;
                        $flagged = true;
                    } else {
                        $categories[$category] = false;
                    }
                } else {
                    // Se não houver threshold configurado, manter valor original da API
                    // Mas garantir que a categoria exista no array
                    if (!isset($categories[$category])) {
                        // Se a API não marcou como detectado, manter false
                        $categories[$category] = false;
                    } else {
                        // Manter valor original da API
                        $categories[$category] = (bool)$categories[$category];
                        if ($categories[$category]) {
                            $flagged = true;
                        }
                    }
                }
            }

            // Atualizar resultado
            $result['categories'] = $categories;
            $result['flagged'] = $flagged;
        }

        return $apiResult;
    }


    /**
     * Salva ou atualiza um threshold
     * @param string $category Nome da categoria
     * @param float $threshold Valor do threshold (0.0 a 1.0)
     * @param bool $enabled Se está habilitado
     * @param string $description Descrição opcional
     * @return bool Sucesso
     */
    public function saveThreshold($category, $threshold, $enabled = true, $description = null) {
        $conn = $this->db->getConnection();
        
        // Validar threshold
        $threshold = max(0.0, min(1.0, floatval($threshold)));
        
        // Garantir que $enabled seja um inteiro (0 ou 1)
        $enabledInt = ($enabled === true || $enabled === 1 || $enabled === '1' || $enabled === 'true') ? 1 : 0;

        // Verificar se a coluna description existe
        $checkDesc = $conn->query("SHOW COLUMNS FROM moderation_thresholds LIKE 'description'");
        $hasDescription = $checkDesc && $checkDesc->num_rows > 0;
        
        if ($hasDescription) {
            $stmt = $conn->prepare("
                INSERT INTO moderation_thresholds (category_name, threshold_value, enabled, description)
                VALUES (?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                    threshold_value = VALUES(threshold_value),
                    enabled = VALUES(enabled),
                    description = VALUES(description)
            ");
            $stmt->bind_param("sdis", $category, $threshold, $enabledInt, $description);
        } else {
            $stmt = $conn->prepare("
                INSERT INTO moderation_thresholds (category_name, threshold_value, enabled)
                VALUES (?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                    threshold_value = VALUES(threshold_value),
                    enabled = VALUES(enabled)
            ");
            $stmt->bind_param("sdi", $category, $threshold, $enabledInt);
        }
        
        $result = $stmt->execute();
        $stmt->close();

        // Limpar cache
        $this->thresholds = null;

        return $result;
    }

    /**
     * Obtém todos os thresholds
     * @return array Array com todos os thresholds
     */
    public function getAllThresholds() {
        $conn = $this->db->getConnection();
        
        // Verificar se a coluna description existe
        $checkDesc = $conn->query("SHOW COLUMNS FROM moderation_thresholds LIKE 'description'");
        $hasDescription = $checkDesc && $checkDesc->num_rows > 0;
        
        $sql = $hasDescription 
            ? "SELECT category_name, threshold_value, enabled, description FROM moderation_thresholds ORDER BY category_name"
            : "SELECT category_name, threshold_value, enabled FROM moderation_thresholds ORDER BY category_name";
        
        $result = $conn->query($sql);

        $thresholds = [];
        while ($row = $result->fetch_assoc()) {
            $thresholds[] = [
                'category' => $row['category_name'],
                'threshold' => floatval($row['threshold_value']),
                'enabled' => (bool)$row['enabled'],
                'description' => $hasDescription ? ($row['description'] ?? '') : ''
            ];
        }

        return $thresholds;
    }

    /**
     * Obtém threshold de uma categoria específica
     * @param string $category Nome da categoria
     * @return float|null Valor do threshold ou null se não encontrado
     */
    public function getThreshold($category) {
        $thresholds = $this->loadThresholds();
        return $thresholds[$category] ?? null;
    }
}

