<?php
/**
 * Script de Instalação
 * Executa o schema do banco de dados automaticamente
 */

require_once __DIR__ . '/config/database.php';

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalação - Sistema de Moderação</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body>
    <div class="container mt-5">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">Instalação do Sistema de Moderação</h4>
                    </div>
                    <div class="card-body">
                        <?php
                        try {
                            $db = Database::getInstance();
                            $conn = $db->getConnection();

                            // Ler arquivo SQL principal
                            $sqlFile = __DIR__ . '/database/schema.sql';
                            if (!file_exists($sqlFile)) {
                                throw new Exception('Arquivo schema.sql não encontrado');
                            }

                            $sql = file_get_contents($sqlFile);
                            
                            // Remover comentários e dividir em comandos
                            $sql = preg_replace('/--.*$/m', '', $sql);
                            $commands = array_filter(array_map('trim', explode(';', $sql)));

                            $success = 0;
                            $errors = [];

                            foreach ($commands as $command) {
                                if (empty($command)) continue;
                                
                                try {
                                    if ($conn->query($command)) {
                                        $success++;
                                    } else {
                                        $errors[] = $conn->error;
                                    }
                                } catch (Exception $e) {
                                    // Ignorar erros de "já existe"
                                    if (strpos($e->getMessage(), 'already exists') === false) {
                                        $errors[] = $e->getMessage();
                                    }
                                }
                            }

                            // Instalar tabela de thresholds
                            $thresholdsFile = __DIR__ . '/database/thresholds_table.sql';
                            if (file_exists($thresholdsFile)) {
                                $thresholdsSql = file_get_contents($thresholdsFile);
                                $thresholdsSql = preg_replace('/--.*$/m', '', $thresholdsSql);
                                $thresholdsCommands = array_filter(array_map('trim', explode(';', $thresholdsSql)));
                                
                                foreach ($thresholdsCommands as $command) {
                                    if (empty($command)) continue;
                                    
                                    try {
                                        if ($conn->query($command)) {
                                            $success++;
                                        }
                                    } catch (Exception $e) {
                                        // Ignorar erros de "já existe"
                                        if (strpos($e->getMessage(), 'already exists') === false && 
                                            strpos($e->getMessage(), 'Duplicate entry') === false) {
                                            // Não adicionar ao array de erros, apenas logar
                                        }
                                    }
                                }
                            }

                            if (empty($errors)) {
                                echo '<div class="alert alert-success">';
                                echo '<h5><i class="bi bi-check-circle"></i> Instalação Concluída!</h5>';
                                echo '<p>O banco de dados foi criado com sucesso.</p>';
                                echo '<p><strong>Comandos executados:</strong> ' . $success . '</p>';
                                echo '</div>';

                                // Verificar tabelas
                                $tables = ['comments', 'moderation_categories', 'feedback', 'moderation_history', 'moderation_stats'];
                                $existingTables = [];
                                
                                foreach ($tables as $table) {
                                    $result = $conn->query("SHOW TABLES LIKE '$table'");
                                    if ($result->num_rows > 0) {
                                        $existingTables[] = $table;
                                    }
                                }

                                echo '<div class="alert alert-info">';
                                echo '<h6>Tabelas criadas:</h6><ul>';
                                foreach ($existingTables as $table) {
                                    echo '<li>' . $table . '</li>';
                                }
                                echo '</ul></div>';

                                echo '<div class="alert alert-warning">';
                                echo '<h6>Próximos passos:</h6>';
                                echo '<ol>';
                                echo '<li>Configure sua chave da API OpenAI em <code>config/openai.php</code></li>';
                                echo '<li>Acesse <a href="index.php">index.php</a> para começar a usar o sistema</li>';
                                echo '</ol>';
                                echo '</div>';

                                echo '<a href="index.php" class="btn btn-primary">Ir para o Sistema</a>';
                            } else {
                                throw new Exception('Erros encontrados: ' . implode(', ', $errors));
                            }

                        } catch (Exception $e) {
                            echo '<div class="alert alert-danger">';
                            echo '<h5><i class="bi bi-exclamation-triangle"></i> Erro na Instalação</h5>';
                            echo '<p>' . htmlspecialchars($e->getMessage()) . '</p>';
                            echo '</div>';
                            echo '<p>Verifique:</p>';
                            echo '<ul>';
                            echo '<li>Se o MySQL está rodando no Laragon</li>';
                            echo '<li>Se as credenciais em <code>config/database.php</code> estão corretas</li>';
                            echo '<li>Se o arquivo <code>database/schema.sql</code> existe</li>';
                            echo '</ul>';
                        }
                        ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>

