<?php
require_once 'config.php';
require_once 'config_ia.php';

header('Content-Type: application/json');

// Timeout global otimizado para velocidade máxima
set_time_limit($PROCESSING_CONFIG['global_timeout']);

// Config IA
$PERSPECTIVE_API_KEY_DEFAULT = "AIzaSyD_KQ77gkUJiGMGBqglxtg9vl5vwoJyTyM";
$PERSPECTIVE_API_KEY = getenv('PERSPECTIVE_API_KEY') ? getenv('PERSPECTIVE_API_KEY') : $PERSPECTIVE_API_KEY_DEFAULT;
$PERSPECTIVE_API_URL = "https://commentanalyzer.googleapis.com/v1alpha1/comments:analyze";
$PERSPECTIVE_LANGUAGES = array_map('trim', explode(',', getenv('PERSPECTIVE_LANGS') ? getenv('PERSPECTIVE_LANGS') : 'pt,en,es,fr'));

// Blacklist desativada: a classificação será 100% pela IA
$HARD_BLACKLIST = [];

function contains_hard_blacklist($text, $blacklist) {
    $t = mb_strtolower($text, 'UTF-8');
    foreach ($blacklist as $w) {
        $w = mb_strtolower($w, 'UTF-8');
        // palavra/frase com bordas de palavra quando possível
        $pattern = '/\b' . preg_quote($w, '/') . '\b/u';
        if (preg_match($pattern, $t)) { return $w; }
        // fallback simples
        if (mb_strpos($t, $w, 0, 'UTF-8') !== false) { return $w; }
    }
    return '';
}

function perspective_call_language($text, $lang, $apiKey, $apiUrl, $timeoutSec = null, $retries = 0) {
    global $TIMEOUT_CONFIG;
    if ($timeoutSec === null) {
        $timeoutSec = $TIMEOUT_CONFIG['main_timeout'];
    }
    $payload = [
        'comment' => ['text' => $text],
        'requestedAttributes' => [
            'TOXICITY' => new stdClass(),
            'SEVERE_TOXICITY' => new stdClass(),
            'INSULT' => new stdClass(),
            'PROFANITY' => new stdClass(),
            'THREAT' => new stdClass(),
            'IDENTITY_ATTACK' => new stdClass(),
        ],
        'doNotStore' => true,
        'languages' => [$lang],
        'spanAnnotations' => false,
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl . '?key=' . urlencode($apiKey));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_TIMEOUT, $timeoutSec);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $TIMEOUT_CONFIG['connect_timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
    curl_setopt($ch, CURLOPT_MAXREDIRS, 0);
    
    $resp = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($code === 200 && $resp) {
        $data = json_decode($resp, true);
        if (isset($data['attributeScores']) && !empty($data['attributeScores'])) {
            return $data['attributeScores'];
        }
    }
    
    return [];
}

function load_blacklist_file($path) {
    if (!file_exists($path)) return [];
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    return array_map(function($l){ return trim($l); }, $lines ?: []);
}

function normalize_text_variant($text) {
    $lower = mb_strtolower($text, 'UTF-8');
    $trans = @iconv('UTF-8', 'ASCII//TRANSLIT', $lower);
    if ($trans !== false) { $lower = $trans; }
    $map = [ '@'=>'a','4'=>'a','1'=>'i','!'=>'i','3'=>'e','0'=>'o','5'=>'s','$'=>'s','7'=>'t','+'=>'t' ];
    $lower = strtr($lower, $map);
    $lower = preg_replace('/[^a-z0-9]+/',' ', $lower);
    $lower = trim(preg_replace('/\s+/',' ', $lower));
    return $lower;
}

function try_fallback_analysis($text, $apiKey, $apiUrl, $thresholds) {
    global $TIMEOUT_CONFIG;
    // Tentar com inglês como fallback mais robusto
    $payload = [
        'comment' => ['text' => $text],
        'requestedAttributes' => [
            'TOXICITY' => new stdClass(),
            'SEVERE_TOXICITY' => new stdClass(),
            'INSULT' => new stdClass(),
            'PROFANITY' => new stdClass(),
            'THREAT' => new stdClass(),
            'IDENTITY_ATTACK' => new stdClass(),
        ],
        'doNotStore' => true,
        'languages' => ['en'],
        'spanAnnotations' => false,
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl . '?key=' . urlencode($apiKey));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_TIMEOUT, $TIMEOUT_CONFIG['fallback_timeout']);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $TIMEOUT_CONFIG['fallback_connect_timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
    curl_setopt($ch, CURLOPT_MAXREDIRS, 0);
    
    $resp = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($code === 200 && $resp) {
        $data = json_decode($resp, true);
        if (isset($data['attributeScores']) && !empty($data['attributeScores'])) {
            $combined = [];
            foreach ($data['attributeScores'] as $attr => $detail) {
                $val = isset($detail['summaryScore']['value']) ? floatval($detail['summaryScore']['value']) : 0.0;
                $combined[$attr] = $val;
            }
            return analyze_one_fast($text, $combined, $thresholds);
        }
    }
    
    // Se fallback falhou, fazer análise básica por palavras-chave
    return analyze_by_keywords($text);
}

function analyze_by_keywords($text) {
    // Fallback simples quando API falha - retorna bom por padrão
    return ['color' => 'risk-green', 'label' => 'Bom', 'detail' => 'Fallback - assumindo positivo'];
}

function analyze_one_fast($text, $combined, $thresholds) {
    // Versão otimizada que usa dados já coletados - mais sensível
    if (empty($combined)) {
        return ['color' => 'risk-neutral', 'label' => 'Erro na análise', 'detail' => 'API não respondeu'];
    }
    
    // Atributo com maior score absoluto
    $worstAttr = '';
    $worstScore = -1.0;
    foreach ($combined as $attr => $val) {
        if ($val > $worstScore) { $worstScore = $val; $worstAttr = $attr; }
    }

    // Razões score/threshold e flags de alto risco
    $ratios = [];
    $worstRatio = 0.0;
    foreach ($thresholds as $attr => $th) {
        $val = isset($combined[$attr]) ? $combined[$attr] : 0.0;
        $ratio = $th > 0 ? ($val / $th) : 0.0;
        $ratios[$attr] = $ratio;
        if ($ratio > $worstRatio) { $worstRatio = $ratio; }
    }
    
    // Calcular percentuais baseados no score real
    $totalScore = array_sum($combined);
    $maxPossibleScore = count($combined) * 1.0; // Máximo possível
    $safety = 1.0 - ($totalScore / $maxPossibleScore);
    $goodPct = (int)round($safety * 100);
    $badPct = 100 - $goodPct;

    // Cor/decisão final baseada 100% na API da Perspective
    if ($worstRatio >= 1.0) {
        $color = 'risk-red';
        $goodPct = 0; $badPct = 100;
    } else if ($worstRatio >= 0.5) {
        $color = 'risk-orange';
    } else if ($worstRatio >= 0.3) {
        $color = 'risk-yellow';
    } else if ($worstRatio >= 0.1) {
        $color = 'risk-blue';
    } else if ($worstScore >= 0.05) {
        // Scores baixos mas significativos = bom
        $color = 'risk-green';
    } else {
        // Scores muito baixos = bom (não neutro)
        $color = 'risk-green';
    }

    // Montar rótulo: motivo quando houver risco (>=0.2)
    global $ATTR_PT;
    $motivo = isset($ATTR_PT[$worstAttr]) ? $ATTR_PT[$worstAttr] : $worstAttr;
    
    // Classificação específica por tipo de atributo
    if ($color === 'risk-red' || $color === 'risk-orange' || $color === 'risk-yellow') {
        if ($worstAttr === 'PROFANITY') {
            $label = 'Ruim - Linguagem imprópria ' . $badPct . '%';
        } elseif ($worstAttr === 'INSULT') {
            $label = 'Ruim - Insulto ' . $badPct . '%';
        } elseif ($worstAttr === 'THREAT') {
            $label = 'Ruim - Ameaça ' . $badPct . '%';
        } elseif ($worstAttr === 'TOXICITY' || $worstAttr === 'SEVERE_TOXICITY') {
            $label = 'Ruim - Crítica ' . $badPct . '%';
        } else {
            $label = 'Ruim - ' . $motivo . ' ' . $badPct . '%';
        }
    } else if ($color === 'risk-green') {
        $label = 'Bom ' . $goodPct . '%';
    } else {
        $label = 'Neutro';
    }

    $detail = $worstAttr !== '' ? ($worstAttr . ': ' . number_format(isset($combined[$worstAttr])?$combined[$worstAttr]:0, 2)) : '';
    return ['color' => $color, 'label' => $label, 'detail' => $detail];
}

function analyze_one($text, $languages, $apiKey, $apiUrl, $thresholds) {
    // Somente IA (Perspective) decide; nenhuma lista rígida
    $combined = [];
    
    // Tentar apenas com português primeiro
    $scores = perspective_call_language($text, 'pt', $apiKey, $apiUrl, 1, 0);
    foreach ($scores as $attr => $detail) {
        $val = isset($detail['summaryScore']['value']) ? floatval($detail['summaryScore']['value']) : 0.0;
        $combined[$attr] = $val;
    }
    
    // Se não funcionou, tentar com inglês como fallback
    if (empty($combined)) {
        $scores = perspective_call_language($text, 'en', $apiKey, $apiUrl, 1, 0);
        foreach ($scores as $attr => $detail) {
            $val = isset($detail['summaryScore']['value']) ? floatval($detail['summaryScore']['value']) : 0.0;
            $combined[$attr] = $val;
        }
    }
    
    if (empty($combined)) {
        // Log do erro para debug
        error_log("Failed to analyze comment: " . substr($text, 0, 100));
        // Tentar fallback com inglês
        return try_fallback_analysis($text, $apiKey, $apiUrl, $thresholds);
    }
    // Atributo com maior score absoluto
    $worstAttr = '';
    $worstScore = -1.0;
    foreach ($combined as $attr => $val) {
        if ($val > $worstScore) { $worstScore = $val; $worstAttr = $attr; }
    }

    // Razões score/threshold e flags de alto risco
    $ratios = [];
    $worstRatio = 0.0;
    foreach ($thresholds as $attr => $th) {
        $val = isset($combined[$attr]) ? $combined[$attr] : 0.0;
        $ratio = $th > 0 ? ($val / $th) : 0.0;
        $ratios[$attr] = $ratio;
        if ($ratio > $worstRatio) { $worstRatio = $ratio; }
    }
    $strongAttrs = ['SEVERE_TOXICITY','INSULT','PROFANITY','THREAT','IDENTITY_ATTACK'];
    $anyStrongHit = false;
    foreach ($strongAttrs as $sa) {
        if (isset($ratios[$sa]) && $ratios[$sa] >= 1.0) { $anyStrongHit = true; break; }
    }
    // Segurança rígida baseada na pior razão (qualquer threshold atingido = Ruim)
    $ratioClamped = min(1.2, max(0.0, $worstRatio));
    $safety = 1.0 - min(1.0, $ratioClamped);
    $goodPct = (int)round($safety * 100);
    $badPct = 100 - $goodPct;

    // Cor/decisão final baseada 100% na API da Perspective
    if ($worstRatio >= 1.0) {
        $color = 'risk-red';
        $goodPct = 0; $badPct = 100;
    } else if ($worstRatio >= 0.5) {
        $color = 'risk-orange';
    } else if ($worstRatio >= 0.3) {
        $color = 'risk-yellow';
    } else if ($worstRatio >= 0.1) {
        $color = 'risk-blue';
    } else {
        // Qualquer coisa abaixo de 0.1 = bom (não neutro)
        $color = 'risk-green';
    }

    // Montar rótulo: motivo quando houver risco (>=0.6)
    global $ATTR_PT;
    $motivo = isset($ATTR_PT[$worstAttr]) ? $ATTR_PT[$worstAttr] : $worstAttr;
    
    // Classificação específica por tipo de atributo
    if ($color === 'risk-red' || $color === 'risk-orange' || $color === 'risk-yellow') {
        if ($worstAttr === 'PROFANITY') {
            $label = 'Ruim - Linguagem imprópria ' . $badPct . '%';
        } elseif ($worstAttr === 'INSULT') {
            $label = 'Ruim - Insulto ' . $badPct . '%';
        } elseif ($worstAttr === 'THREAT') {
            $label = 'Ruim - Ameaça ' . $badPct . '%';
        } elseif ($worstAttr === 'TOXICITY' || $worstAttr === 'SEVERE_TOXICITY') {
            $label = 'Ruim - Crítica ' . $badPct . '%';
        } else {
            $label = 'Ruim - ' . $motivo . ' ' . $badPct . '%';
        }
    } else {
        $label = 'Bom ' . $goodPct . '%';
    }

    $detail = $worstAttr !== '' ? ($worstAttr . ': ' . number_format(isset($combined[$worstAttr])?$combined[$worstAttr]:0, 2)) : '';
    return ['color' => $color, 'label' => $label, 'detail' => $detail];
}

// Lê JSON
$raw = file_get_contents('php://input');
$json = json_decode($raw, true);

// Suporta modo single (comment) ou batch (comments)
if (isset($json['comment']) && is_string($json['comment'])) {
    $res = analyze_one($json['comment'], $PERSPECTIVE_LANGUAGES, $PERSPECTIVE_API_KEY, $PERSPECTIVE_API_URL, $AI_THRESHOLDS);
    echo json_encode(['result' => $res]);
    exit;
}

$comments = isset($json['comments']) && is_array($json['comments']) ? $json['comments'] : [];
$results = [];

// Log de início para debug
error_log("Iniciando análise de " . count($comments) . " comentários");

// Processamento em tempo real - retorna resultados assim que ficam prontos
$multiHandle = curl_multi_init();
$curlHandles = [];
$completed = [];
$maxConcurrent = min($PROCESSING_CONFIG['max_concurrent'], count($comments));

// Preparar TODAS as requisições de uma vez
foreach ($comments as $i => $comment) {
    // Validar e limpar comentário
    $comment = trim($comment);
    if (empty($comment) || strlen($comment) > 2000) {
        $completed[$i] = ['color' => 'risk-blue', 'label' => 'Neutro', 'detail' => 'Comentário inválido'];
        continue;
    }
    
    $payload = [
        'comment' => ['text' => $comment],
        'requestedAttributes' => [
            'TOXICITY' => new stdClass(),
            'SEVERE_TOXICITY' => new stdClass(),
            'INSULT' => new stdClass(),
            'PROFANITY' => new stdClass(),
            'THREAT' => new stdClass(),
            'IDENTITY_ATTACK' => new stdClass(),
        ],
        'doNotStore' => true,
        'languages' => ['pt'],
        'spanAnnotations' => false,
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $PERSPECTIVE_API_URL . '?key=' . urlencode($PERSPECTIVE_API_KEY));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_TIMEOUT, $TIMEOUT_CONFIG['main_timeout']);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $TIMEOUT_CONFIG['connect_timeout']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
    curl_setopt($ch, CURLOPT_MAXREDIRS, 0);
    
    $curlHandles[$i] = $ch;
    curl_multi_add_handle($multiHandle, $ch);
}

// Processar resultados conforme ficam prontos
$running = null;
$processed = 0;
$totalComments = count($comments);

do {
    curl_multi_exec($multiHandle, $running);
    
    // Verificar se alguma requisição terminou
    while ($info = curl_multi_info_read($multiHandle)) {
        if ($info['msg'] == CURLMSG_DONE) {
            $ch = $info['handle'];
            $index = array_search($ch, $curlHandles);
            
            if ($index !== false) {
                $comment = $comments[$index];
                $resp = curl_multi_getcontent($ch);
                $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                
                if ($code === 200 && $resp) {
                    $data = json_decode($resp, true);
                    if (isset($data['attributeScores']) && !empty($data['attributeScores'])) {
                        $combined = [];
                        foreach ($data['attributeScores'] as $attr => $detail) {
                            $val = isset($detail['summaryScore']['value']) ? floatval($detail['summaryScore']['value']) : 0.0;
                            $combined[$attr] = $val;
                        }
                        $completed[$index] = analyze_one_fast($comment, $combined, $AI_THRESHOLDS);
                    } else {
                        // Fallback com inglês se português não funcionou
                        error_log("API retornou vazio para português, tentando inglês para: " . substr($comment, 0, 50));
                        $completed[$index] = try_fallback_analysis($comment, $PERSPECTIVE_API_KEY, $PERSPECTIVE_API_URL, $AI_THRESHOLDS);
                    }
                } else {
                    // Fallback se a requisição falhou
                    error_log("Requisição falhou (código $code), usando fallback para: " . substr($comment, 0, 50));
                    $completed[$index] = try_fallback_analysis($comment, $PERSPECTIVE_API_KEY, $PERSPECTIVE_API_URL, $AI_THRESHOLDS);
                }
                
                curl_multi_remove_handle($multiHandle, $ch);
                curl_close($ch);
                $processed++;
                
                // Log do progresso
                error_log("Processado comentário $index: " . substr($comment, 0, 50) . "...");
            }
        }
    }
    
    curl_multi_select($multiHandle);
} while ($running > 0);

curl_multi_close($multiHandle);

// Ordenar resultados pela ordem original
$results = [];
for ($i = 0; $i < $totalComments; $i++) {
    if (isset($completed[$i])) {
        $results[] = $completed[$i];
    }
}

error_log("Análise concluída: " . count($results) . " resultados gerados");

echo json_encode([
    'status' => 'complete',
    'processed' => count($results),
    'total' => $totalComments,
    'results' => $results
]);
exit;
?>

