<?php
/**
 * Script para verificar se todas as tabelas necessárias existem
 */

require_once 'config.php';

echo "<h1>Verificação do Banco de Dados</h1>";
echo "<style>
    body { font-family: Arial; padding: 20px; }
    .success { color: green; }
    .error { color: red; }
    .info { color: blue; }
    table { border-collapse: collapse; margin: 20px 0; }
    td, th { border: 1px solid #ccc; padding: 8px; text-align: left; }
    th { background: #f0f0f0; }
</style>";

// Listar tabelas necessárias
$requiredTables = [
    'comment_blacklist',
    'comments_ia', // Tabela específica para IA (criada se comments já existir)
    'moderation_categories',
    'moderation_history',
    'moderation_thresholds'
];

// Verificar também se comments existe
$checkComments = $conn->query("SHOW TABLES LIKE 'comments'");
if ($checkComments && $checkComments->num_rows > 0) {
    echo "<p class='info'>ℹ Tabela 'comments' encontrada (tabela principal do sistema)</p>";
}

echo "<h2>1. Verificando conexão com o banco</h2>";
if ($conn->connect_error) {
    echo "<p class='error'>✗ Erro de conexão: " . $conn->connect_error . "</p>";
    exit;
}
echo "<p class='success'>✓ Conectado com sucesso ao banco: " . DB_NAME . "</p>";

echo "<h2>2. Verificando tabelas necessárias</h2>";
echo "<table>";
echo "<tr><th>Tabela</th><th>Status</th><th>Registros</th></tr>";

foreach ($requiredTables as $table) {
    $exists = false;
    $count = 0;
    
    // Verificar se tabela existe
    $result = $conn->query("SHOW TABLES LIKE '$table'");
    if ($result && $result->num_rows > 0) {
        $exists = true;
        
        // Contar registros
        $countResult = $conn->query("SELECT COUNT(*) as cnt FROM `$table`");
        if ($countResult) {
            $row = $countResult->fetch_assoc();
            $count = $row['cnt'];
        }
    }
    
    echo "<tr>";
    echo "<td><code>$table</code></td>";
    echo "<td class='" . ($exists ? 'success' : 'error') . "'>" . ($exists ? '✓ Existe' : '✗ Não existe') . "</td>";
    echo "<td>" . ($exists ? $count : '-') . "</td>";
    echo "</tr>";
}

echo "</table>";

echo "<h2>3. Verificando estrutura das tabelas</h2>";

// Verificar tabela comments
echo "<h3>Tabela: comments</h3>";
$result = $conn->query("SHOW COLUMNS FROM comments");
if ($result) {
    echo "<table>";
    echo "<tr><th>Coluna</th><th>Tipo</th><th>Null</th><th>Key</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>{$row['Field']}</td>";
        echo "<td>{$row['Type']}</td>";
        echo "<td>{$row['Null']}</td>";
        echo "<td>{$row['Key']}</td>";
        echo "</tr>";
    }
    echo "</table>";
} else {
    echo "<p class='error'>✗ Erro ao verificar estrutura: " . $conn->error . "</p>";
}

// Verificar tabela moderation_thresholds
echo "<h3>Tabela: moderation_thresholds</h3>";
$result = $conn->query("SHOW COLUMNS FROM moderation_thresholds");
if ($result) {
    echo "<table>";
    echo "<tr><th>Coluna</th><th>Tipo</th><th>Null</th><th>Key</th></tr>";
    while ($row = $result->fetch_assoc()) {
        echo "<tr>";
        echo "<td>{$row['Field']}</td>";
        echo "<td>{$row['Type']}</td>";
        echo "<td>{$row['Null']}</td>";
        echo "<td>{$row['Key']}</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    // Listar thresholds configurados
    $result = $conn->query("SELECT * FROM moderation_thresholds");
    if ($result && $result->num_rows > 0) {
        echo "<h4>Thresholds Configurados:</h4>";
        echo "<table>";
        echo "<tr><th>Categoria</th><th>Threshold</th><th>Ativo</th></tr>";
        while ($row = $result->fetch_assoc()) {
            echo "<tr>";
            echo "<td>{$row['category_name']}</td>";
            echo "<td>" . number_format($row['threshold_value'], 2) . "</td>";
            echo "<td>" . ($row['enabled'] ? '✓' : '✗') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<p class='info'>ℹ Nenhum threshold configurado (será usado padrão da OpenAI)</p>";
    }
} else {
    echo "<p class='error'>✗ Erro ao verificar estrutura: " . $conn->error . "</p>";
}

echo "<h2>4. Verificando chave da API OpenAI</h2>";
require_once 'IA/config/openai.php';

if (defined('OPENAI_API_KEY') && !empty(OPENAI_API_KEY)) {
    $key = OPENAI_API_KEY;
    $keyPreview = substr($key, 0, 10) . '...' . substr($key, -4);
    echo "<p class='success'>✓ Chave configurada: <code>$keyPreview</code></p>";
    
    // Testar chave
    echo "<p class='info'>Testando chave da API...</p>";
    
    $ch = curl_init('https://api.openai.com/v1/moderations');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . $key
    ]);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
        'input' => 'teste',
        'model' => OPENAI_MODEL
    ]));
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        echo "<p class='error'>✗ Erro de conexão: $error</p>";
    } elseif ($httpCode === 200) {
        echo "<p class='success'>✓ Chave válida! API respondeu com sucesso.</p>";
    } elseif ($httpCode === 401) {
        echo "<p class='error'>✗ Chave inválida (HTTP 401)</p>";
    } else {
        $responseData = json_decode($response, true);
        $errorMsg = $responseData['error']['message'] ?? $response;
        echo "<p class='error'>✗ Erro HTTP $httpCode: " . htmlspecialchars($errorMsg) . "</p>";
    }
} else {
    echo "<p class='error'>✗ Chave da API não configurada</p>";
}

echo "<h2>5. Criação automática de tabelas faltantes</h2>";

// Verificar e criar comments_ia se não existir
$checkCommentsIA = $conn->query("SHOW TABLES LIKE 'comments_ia'");
if (!$checkCommentsIA || $checkCommentsIA->num_rows === 0) {
    echo "<p class='warning'>⚠ Tabela comments_ia não encontrada. Criando...</p>";
    
    $sqlCreateCommentsIA = "CREATE TABLE IF NOT EXISTS `comments_ia` (
        `id` INT AUTO_INCREMENT PRIMARY KEY,
        `content` TEXT NOT NULL,
        `flagged` BOOLEAN DEFAULT FALSE,
        `moderation_id` VARCHAR(255),
        `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX `idx_flagged` (`flagged`),
        INDEX `idx_created_at` (`created_at`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    if ($conn->query($sqlCreateCommentsIA)) {
        echo "<p class='success'>✓ Tabela comments_ia criada com sucesso!</p>";
    } else {
        echo "<p class='error'>✗ Erro ao criar comments_ia: " . $conn->error . "</p>";
    }
} else {
    echo "<p class='success'>✓ Tabela comments_ia já existe</p>";
}

echo "<h2>6. Teste completo do fluxo</h2>";
echo "<p>Testando envio de um comentário simples...</p>";

try {
    // Simular requisição para ia_analyze_openai.php
    $testComment = "Este é um teste de comentário";
    
    echo "<p>Enviando: <code>$testComment</code></p>";
    
    $ch = curl_init('http://localhost/moderacao/comentario/ia_analyze_openai.php');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['comments' => [$testComment]]));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        echo "<p class='error'>✗ Erro de conexão: $error</p>";
    } elseif ($httpCode === 200) {
        echo "<p class='success'>✓ Sucesso! API respondeu com HTTP 200</p>";
        $data = json_decode($response, true);
        echo "<pre>" . htmlspecialchars(json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) . "</pre>";
    } else {
        echo "<p class='error'>✗ Erro HTTP $httpCode</p>";
        echo "<p>Resposta:</p>";
        echo "<pre>" . htmlspecialchars($response) . "</pre>";
    }
    
} catch (Exception $e) {
    echo "<p class='error'>✗ Exceção: " . $e->getMessage() . "</p>";
}

echo "<hr>";
echo "<p><a href='test_ia_api.html'>Abrir página de testes interativos</a></p>";
echo "<p><a href='install.php'>Executar instalação/verificação</a></p>";

$conn->close();
?>

