<?php
/**
 * Endpoint de análise de comentários usando OpenAI
 * Integra com IA/api/moderate.php e adapta formato de resposta
 * Verifica blacklist antes de enviar para IA
 */

// Habilitar exibição de erros para debug (desabilitar em produção)
error_reporting(E_ALL);
ini_set('display_errors', 0); // Não exibir na tela
ini_set('log_errors', 1); // Logar erros

// Capturar erros fatais
register_shutdown_function(function() {
    $error = error_get_last();
    if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        header('Content-Type: application/json; charset=utf-8');
        http_response_code(500);
        echo json_encode([
            'error' => 'Erro fatal no servidor',
            'details' => $error['message'],
            'file' => basename($error['file']),
            'line' => $error['line']
        ], JSON_UNESCAPED_UNICODE);
    }
});

header('Content-Type: application/json; charset=utf-8');

// Conectar ao banco de dados para verificar blacklist
try {
    require_once 'config.php';
} catch (Exception $e) {
    error_log("ia_analyze_openai.php - Erro ao carregar config.php: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Erro ao carregar configuração'], JSON_UNESCAPED_UNICODE);
    exit;
}

// Verificar conexão com banco
if (!isset($conn) || $conn->connect_error) {
    error_log("ia_analyze_openai.php - Erro de conexão com banco de dados");
    http_response_code(500);
    echo json_encode(['error' => 'Erro de conexão com banco de dados'], JSON_UNESCAPED_UNICODE);
    exit;
}

// Log de início
error_log("ia_analyze_openai.php - Requisição recebida - Método: " . $_SERVER['REQUEST_METHOD']);

// Verificar se é POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método não permitido. Use POST.'], JSON_UNESCAPED_UNICODE);
    exit;
}

// Ler dados JSON
$raw = file_get_contents('php://input');
$data = json_decode($raw, true);

// Log de debug (remover em produção)
error_log("ia_analyze_openai.php - Raw data: " . substr($raw, 0, 200));
error_log("ia_analyze_openai.php - Decoded data: " . print_r($data, true));

// Verificar se JSON é válido
if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    echo json_encode([
        'error' => 'JSON inválido: ' . json_last_error_msg(),
        'received' => substr($raw, 0, 100)
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Suporta formato antigo (comments) e novo (input)
$comments = [];
if (isset($data['comments']) && is_array($data['comments'])) {
    $comments = $data['comments'];
} elseif (isset($data['comment']) && is_string($data['comment'])) {
    $comments = [$data['comment']];
} elseif (isset($data['input'])) {
    if (is_array($data['input'])) {
        $comments = $data['input'];
    } else {
        $comments = [$data['input']];
    }
} else {
    http_response_code(400);
    echo json_encode([
        'error' => 'Dados inválidos. Envie "comments" (array) ou "input" (string ou array)',
        'received_keys' => array_keys($data ?: []),
        'data_type' => gettype($data)
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

if (empty($comments)) {
    http_response_code(400);
    echo json_encode([
        'error' => 'Nenhum comentário fornecido',
        'comments_count' => count($comments),
        'data_received' => $data
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Função para verificar se um comentário está na blacklist
function isBlacklisted($conn, $comment) {
    $commentTrimmed = trim($comment);
    if (empty($commentTrimmed)) {
        return false;
    }
    
    // Buscar na blacklist (comparação case-insensitive)
    $stmt = $conn->prepare("SELECT id FROM comment_blacklist WHERE LOWER(TRIM(phrase)) = LOWER(?) LIMIT 1");
    if (!$stmt) {
        return false;
    }
    
    $stmt->bind_param("s", $commentTrimmed);
    $stmt->execute();
    $result = $stmt->get_result();
    $isBlacklisted = $result->num_rows > 0;
    $stmt->close();
    
    return $isBlacklisted;
}

// Verificar blacklist para cada comentário
$blacklistResults = [];
$commentsToAnalyze = [];
$commentsIndexMap = []; // Mapeia índice original para novo índice

foreach ($comments as $index => $comment) {
    if (isBlacklisted($conn, $comment)) {
        // Comentário está na blacklist - retornar como bloqueado
        $blacklistResults[$index] = [
            'color' => 'risk-red',
            'label' => 'Bloqueado - Blacklist',
            'detail' => 'Este comentário está na lista negra',
            'maxScore' => 1.0,
            'maxAttr' => 'blacklist'
        ];
    } else {
        // Comentário não está na blacklist - adicionar para análise pela IA
        $commentsIndexMap[count($commentsToAnalyze)] = $index;
        $commentsToAnalyze[] = $comment;
    }
}

// Se todos os comentários estão na blacklist, retornar apenas resultados da blacklist
if (empty($commentsToAnalyze)) {
    $finalResults = [];
    for ($i = 0; $i < count($comments); $i++) {
        $finalResults[] = $blacklistResults[$i] ?? [
            'color' => 'risk-neutral',
            'label' => 'Erro',
            'detail' => ''
        ];
    }
    
    echo json_encode([
        'status' => 'complete',
        'processed' => count($finalResults),
        'total' => count($comments),
        'results' => $finalResults
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Chamar API da pasta IA apenas para comentários não bloqueados
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'];
$basePath = dirname($_SERVER['SCRIPT_NAME']); // comentario/
$apiUrl = $protocol . '://' . $host . $basePath . '/IA/api/moderate.php';

error_log("ia_analyze_openai.php - Enviando " . count($commentsToAnalyze) . " comentários para API OpenAI");
error_log("ia_analyze_openai.php - URL da API: " . $apiUrl);

$startTime = microtime(true);

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $apiUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
// Usar modo rápido (skip_save=true) para não salvar no banco
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
    'input' => $commentsToAnalyze,
    'skip_save' => true
]));
curl_setopt($ch, CURLOPT_TIMEOUT, 60);
curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$curlError = curl_error($ch);
curl_close($ch);

$elapsed = round((microtime(true) - $startTime), 2);
error_log("ia_analyze_openai.php - Resposta recebida em {$elapsed}s - HTTP {$httpCode}");

if ($curlError) {
    http_response_code(500);
    // Mesclar resultados da blacklist com erros
    $finalResults = [];
    for ($i = 0; $i < count($comments); $i++) {
        if (isset($blacklistResults[$i])) {
            $finalResults[] = $blacklistResults[$i];
        } else {
            $finalResults[] = [
                'color' => 'risk-neutral',
                'label' => 'Erro de conexão',
                'detail' => ''
            ];
        }
    }
    echo json_encode([
        'error' => 'Erro de conexão: ' . $curlError,
        'results' => $finalResults
    ]);
    exit;
}

if ($httpCode !== 200 || !$response) {
    error_log("ia_analyze_openai.php - Erro HTTP da API interna: " . $httpCode);
    error_log("ia_analyze_openai.php - Resposta da API: " . substr($response, 0, 500));
    
    http_response_code($httpCode ?: 500);
    
    // Tentar decodificar resposta de erro
    $errorDetails = null;
    if ($response) {
        $errorData = json_decode($response, true);
        if ($errorData && isset($errorData['error'])) {
            $errorDetails = $errorData['error'];
            if (isset($errorData['error_file'])) {
                $errorDetails .= " (arquivo: " . $errorData['error_file'] . ", linha: " . $errorData['error_line'] . ")";
            }
        }
    }
    
    // Mesclar resultados da blacklist com erros
    $finalResults = [];
    for ($i = 0; $i < count($comments); $i++) {
        if (isset($blacklistResults[$i])) {
            $finalResults[] = $blacklistResults[$i];
        } else {
            $finalResults[] = [
                'color' => 'risk-neutral',
                'label' => 'Erro API',
                'detail' => $errorDetails ?: ''
            ];
        }
    }
    echo json_encode([
        'error' => 'Erro na API: HTTP ' . $httpCode . ($errorDetails ? ' - ' . $errorDetails : ''),
        'results' => $finalResults
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$apiData = json_decode($response, true);

if (!$apiData || !isset($apiData['success']) || !$apiData['success']) {
    http_response_code(500);
    // Mesclar resultados da blacklist com erros
    $finalResults = [];
    for ($i = 0; $i < count($comments); $i++) {
        if (isset($blacklistResults[$i])) {
            $finalResults[] = $blacklistResults[$i];
        } else {
            $finalResults[] = [
                'color' => 'risk-neutral',
                'label' => 'Erro',
                'detail' => ''
            ];
        }
    }
    echo json_encode([
        'error' => $apiData['error'] ?? 'Erro desconhecido',
        'results' => $finalResults
    ]);
    exit;
}

// Processar resultados da IA e converter para formato compatível
$iaResults = [];
$apiResults = is_array($apiData['results']) ? $apiData['results'] : [$apiData['results']];

foreach ($apiResults as $index => $result) {
    $flagged = $result['flagged'] ?? false;
    $categories = $result['categories'] ?? [];
    $scores = $result['category_scores'] ?? [];
    
    // Encontrar categoria com maior score
    $maxScore = 0.0;
    $maxCategory = '';
    foreach ($scores as $cat => $score) {
        $score = floatval($score);
        if ($score > $maxScore) {
            $maxScore = $score;
            $maxCategory = $cat;
        }
    }
    
    // Mapear categorias OpenAI para português
    $categoryLabels = [
        'hate' => 'Ódio',
        'hate/threatening' => 'Ódio/Ameaçador',
        'harassment' => 'Assédio',
        'harassment/threatening' => 'Assédio/Ameaçador',
        'self-harm' => 'Autolesão',
        'self-harm/intent' => 'Autolesão/Intenção',
        'self-harm/instructions' => 'Autolesão/Instruções',
        'sexual' => 'Sexual',
        'sexual/minors' => 'Sexual/Menores',
        'violence' => 'Violência',
        'violence/graphic' => 'Violência/Gráfico',
        'illicit' => 'Ilícito',
        'illicit/violent' => 'Ilícito/Violento'
    ];
    
    // Determinar cor e label baseado em flagged e scores
    if ($flagged) {
        if ($maxScore >= 0.7) {
            $color = 'risk-red';
        } elseif ($maxScore >= 0.4) {
            $color = 'risk-orange';
        } else {
            $color = 'risk-yellow';
        }
        
        $label = 'Ruim - ' . ($categoryLabels[$maxCategory] ?? $maxCategory) . ' ' . round($maxScore * 100) . '%';
    } else {
        $color = 'risk-green';
        $label = 'Bom ' . round((1 - $maxScore) * 100) . '%';
    }
    
    // Montar detalhe
    $detail = $maxCategory ? ($maxCategory . ': ' . number_format($maxScore, 2)) : '';
    
    $iaResults[] = [
        'color' => $color,
        'label' => $label,
        'detail' => $detail,
        'maxScore' => $maxScore,
        'maxAttr' => $maxCategory,
        'raw' => $result
    ];
}

// Mesclar resultados da blacklist com resultados da IA na ordem original
$finalResults = [];
for ($i = 0; $i < count($comments); $i++) {
    if (isset($blacklistResults[$i])) {
        // Comentário estava na blacklist
        $finalResults[] = $blacklistResults[$i];
    } else {
        // Buscar resultado da IA usando o mapeamento
        $iaIndex = array_search($i, $commentsIndexMap);
        if ($iaIndex !== false && isset($iaResults[$iaIndex])) {
            $finalResults[] = $iaResults[$iaIndex];
        } else {
            // Fallback caso algo dê errado
            $finalResults[] = [
                'color' => 'risk-neutral',
                'label' => 'Erro',
                'detail' => ''
            ];
        }
    }
}

// Retornar no formato esperado pelo JavaScript
echo json_encode([
    'status' => 'complete',
    'processed' => count($finalResults),
    'total' => count($comments),
    'results' => $finalResults
], JSON_UNESCAPED_UNICODE);

