<?php
// Página de cadastro oculta - apenas para administrador
require_once 'config.php';

// Primeiro, criar a tabela se não existir
if ($mysqli) {
    $create_table_sql = "CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        username VARCHAR(50) NOT NULL UNIQUE,
        password VARCHAR(255) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        is_active TINYINT(1) DEFAULT 1,
        INDEX idx_username (username),
        INDEX idx_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $mysqli->query($create_table_sql);
}

// Verificar se existe usuário admin no sistema
$has_admin = false;
if ($mysqli) {
    $admin_username = $_ENV['ADMIN_USERNAME'];
    $check_admin = $mysqli->query("SELECT COUNT(*) as count FROM users WHERE username = '$admin_username'");
    if ($check_admin) {
        $count = $check_admin->fetch_assoc()['count'];
        $has_admin = ($count > 0);
    }
}

// Se não tem admin, redirecionar para setup
if (!$has_admin) {
    header('Location: setup_admin');
    exit;
}

// Verificar se o usuário está logado
if (!isLoggedIn()) {
    header('Location: login');
    exit;
}

// Verificar se é admin - versão simplificada
$is_admin = false;
if (isset($_SESSION['username'])) {
    // Verificar se é admin no banco de dados
    $admin_username = $_ENV['ADMIN_USERNAME'];
    $admin_check = $mysqli->prepare("SELECT id FROM users WHERE username = ?");
    $admin_check->bind_param("s", $_SESSION['username']);
    $admin_check->execute();
    $admin_result = $admin_check->get_result();
    
    if ($admin_result->num_rows > 0) {
        $is_admin = true;
    }
    $admin_check->close();
}

// Se não for admin, redirecionar
if (!$is_admin) {
    echo "<h2>Acesso Negado</h2>";
    echo "<p>Você precisa ser administrador para acessar esta página.</p>";
    echo "<p>Usuário atual: " . ($_SESSION['username'] ?? 'Não logado') . "</p>";
    echo "<a href='index'>Voltar ao sistema</a>";
    exit;
}

// Processar cadastro
$message = '';
$error = '';

if ($_POST) {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    // Validações
    if (empty($username) || empty($password)) {
        $error = 'Todos os campos são obrigatórios.';
    } elseif ($password !== $confirm_password) {
        $error = 'As senhas não coincidem.';
    } elseif (strlen($password) < 6) {
        $error = 'A senha deve ter pelo menos 6 caracteres.';
    } else {
        // Verificar se usuário já existe
        $check_sql = "SELECT id FROM users WHERE username = ?";
        $stmt = $mysqli->prepare($check_sql);
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows > 0) {
            $error = 'Username já existe.';
        } else {
            // Criar usuário
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            $insert_sql = "INSERT INTO users (username, password) VALUES (?, ?)";
            $stmt = $mysqli->prepare($insert_sql);
            $stmt->bind_param("ss", $username, $hashed_password);
            
            if ($stmt->execute()) {
                $message = 'Usuário cadastrado com sucesso!';
                // Limpar formulário
                $username = '';
            } else {
                $error = 'Erro ao cadastrar usuário: ' . $stmt->error;
            }
        }
        $stmt->close();
    }
}

// Listar usuários existentes
$users_sql = "SELECT id, username, created_at, is_active FROM users ORDER BY created_at DESC";
$users_result = $mysqli->query($users_sql);
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cadastro de Usuários - Admin</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2rem;
            margin-bottom: 10px;
        }
        
        .header p {
            opacity: 0.9;
        }
        
        .content {
            padding: 40px;
        }
        
        .form-section {
            background: #f8f9fa;
            padding: 30px;
            border-radius: 10px;
            margin-bottom: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
        }
        
        .form-group input {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e1e5e9;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: #667eea;
        }
        
        .btn {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
        }
        
        .btn:hover {
            transform: translateY(-2px);
        }
        
        .btn-secondary {
            background: #6c757d;
            margin-left: 10px;
        }
        
        .alert {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .alert-success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert-danger {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .users-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        
        .users-table th,
        .users-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #e1e5e9;
        }
        
        .users-table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #333;
        }
        
        .status-active {
            color: #28a745;
            font-weight: 600;
        }
        
        .status-inactive {
            color: #dc3545;
            font-weight: 600;
        }
        
        .back-link {
            display: inline-block;
            margin-bottom: 20px;
            color: #667eea;
            text-decoration: none;
            font-weight: 600;
        }
        
        .back-link:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><i class="fas fa-user-plus"></i> Cadastro de Usuários</h1>
            <p>Página administrativa para gerenciar usuários do sistema</p>
        </div>
        
        <div class="content">
            <a href="index" class="back-link">
                <i class="fas fa-arrow-left"></i> Voltar ao Sistema
            </a>
            
            <?php if ($message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?= $message ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle"></i> <?= $error ?>
                </div>
            <?php endif; ?>
            
            <div class="form-section">
                <h2><i class="fas fa-user-plus"></i> Novo Usuário</h2>
                <form method="POST">
                    <div class="form-group">
                        <label for="username">Username:</label>
                        <input type="text" id="username" name="username" value="<?= htmlspecialchars($username ?? '') ?>" required>
                    </div>
                    
                    
                    <div class="form-group">
                        <label for="password">Senha:</label>
                        <input type="password" id="password" name="password" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="confirm_password">Confirmar Senha:</label>
                        <input type="password" id="confirm_password" name="confirm_password" required>
                    </div>
                    
                    <button type="submit" class="btn">
                        <i class="fas fa-save"></i> Cadastrar Usuário
                    </button>
                </form>
            </div>
            
            <div class="form-section">
                <h2><i class="fas fa-users"></i> Usuários Cadastrados</h2>
                <?php if ($users_result && $users_result->num_rows > 0): ?>
                    <table class="users-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Username</th>
                                <th>Data de Cadastro</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($user = $users_result->fetch_assoc()): ?>
                                <tr>
                                    <td><?= $user['id'] ?></td>
                                    <td><?= htmlspecialchars($user['username']) ?></td>
                                    <td><?= date('d/m/Y H:i', strtotime($user['created_at'])) ?></td>
                                    <td>
                                        <span class="<?= $user['is_active'] ? 'status-active' : 'status-inactive' ?>">
                                            <?= $user['is_active'] ? 'Ativo' : 'Inativo' ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p>Nenhum usuário cadastrado.</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>
