<?php
require_once 'config.php';

header('Content-Type: text/event-stream');
header('Cache-Control: no-cache');
header('Connection: keep-alive');

// Timeout global otimizado para velocidade máxima
set_time_limit(30);

// Config IA
$PERSPECTIVE_API_KEY_DEFAULT = "AIzaSyD_KQ77gkUJiGMGBqglxtg9vl5vwoJyTyM";
$PERSPECTIVE_API_KEY = getenv('PERSPECTIVE_API_KEY') ? getenv('PERSPECTIVE_API_KEY') : $PERSPECTIVE_API_KEY_DEFAULT;
$PERSPECTIVE_API_URL = "https://commentanalyzer.googleapis.com/v1alpha1/comments:analyze";

$AI_THRESHOLDS = [
    'TOXICITY' => 0.05,
    'SEVERE_TOXICITY' => 0.05,
    'INSULT' => 0.05,
    'PROFANITY' => 0.90,
    'THREAT' => 0.05,
    'IDENTITY_ATTACK' => 0.05,
];

$ATTR_PT = [
    'TOXICITY' => 'Critica',
    'SEVERE_TOXICITY' => 'Critica grave',
    'INSULT' => 'Insulto',
    'PROFANITY' => 'Linguagem imprópria',
    'THREAT' => 'Ameaça',
    'IDENTITY_ATTACK' => 'Ataque',
];

function analyze_one_fast($text, $combined, $thresholds) {
    if (empty($combined)) {
        return ['color' => 'risk-neutral', 'label' => 'Erro na análise', 'detail' => 'API não respondeu'];
    }
    
    $worstAttr = '';
    $worstScore = -1.0;
    foreach ($combined as $attr => $val) {
        if ($val > $worstScore) { $worstScore = $val; $worstAttr = $attr; }
    }

    $ratios = [];
    $worstRatio = 0.0;
    foreach ($thresholds as $attr => $th) {
        $val = isset($combined[$attr]) ? $combined[$attr] : 0.0;
        $ratio = $th > 0 ? ($val / $th) : 0.0;
        $ratios[$attr] = $ratio;
        if ($ratio > $worstRatio) { $worstRatio = $ratio; }
    }
    
    $ratioClamped = min(1.2, max(0.0, $worstRatio));
    $safety = 1.0 - min(1.0, $ratioClamped);
    $goodPct = (int)round($safety * 100);
    $badPct = 100 - $goodPct;

    if ($worstRatio >= 1.0) {
        $color = 'risk-red';
        $goodPct = 0; $badPct = 100;
    } else if ($worstRatio >= 0.8) {
        $color = 'risk-orange';
    } else if ($worstRatio >= 0.6) {
        $color = 'risk-yellow';
    } else if ($safety >= 0.9) {
        $color = 'risk-blue';
    } else {
        $color = 'risk-green';
    }

    global $ATTR_PT;
    $motivo = isset($ATTR_PT[$worstAttr]) ? $ATTR_PT[$worstAttr] : $worstAttr;
    if ($color === 'risk-red' || $color === 'risk-orange' || $color === 'risk-yellow') {
        $label = 'Ruim - ' . $motivo . ' ' . $badPct . '%';
    } else {
        $label = 'Bom ' . $goodPct . '%';
    }

    $detail = $worstAttr !== '' ? ($worstAttr . ': ' . number_format(isset($combined[$worstAttr])?$combined[$worstAttr]:0, 2)) : '';
    return ['color' => $color, 'label' => $label, 'detail' => $detail];
}

function sendSSE($data) {
    echo "data: " . json_encode($data) . "\n\n";
    flush();
}

// Lê comentários de GET ou POST
$comments = [];

if (isset($_GET['comments'])) {
    $comments = json_decode($_GET['comments'], true);
} else {
    $raw = file_get_contents('php://input');
    $json = json_decode($raw, true);
    $comments = isset($json['comments']) && is_array($json['comments']) ? $json['comments'] : [];
}

if (empty($comments)) {
    sendSSE(['error' => 'Nenhum comentário fornecido']);
    exit;
}

sendSSE(['status' => 'started', 'total' => count($comments)]);

// Processamento em tempo real com SSE
$multiHandle = curl_multi_init();
$curlHandles = [];
$completed = [];

// Preparar TODAS as requisições de uma vez
foreach ($comments as $i => $comment) {
    $comment = trim($comment);
    if (empty($comment) || strlen($comment) > 2000) {
        $completed[$i] = ['color' => 'risk-blue', 'label' => 'Neutro', 'detail' => 'Comentário inválido'];
        sendSSE(['index' => $i, 'result' => $completed[$i], 'processed' => count($completed)]);
        continue;
    }
    
    $payload = [
        'comment' => ['text' => $comment],
        'requestedAttributes' => [
            'TOXICITY' => new stdClass(),
            'SEVERE_TOXICITY' => new stdClass(),
            'INSULT' => new stdClass(),
            'PROFANITY' => new stdClass(),
            'THREAT' => new stdClass(),
            'IDENTITY_ATTACK' => new stdClass(),
        ],
        'doNotStore' => true,
        'languages' => ['pt'],
        'spanAnnotations' => false,
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $PERSPECTIVE_API_URL . '?key=' . urlencode($PERSPECTIVE_API_KEY));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_TIMEOUT, 1);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 1);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
    curl_setopt($ch, CURLOPT_MAXREDIRS, 0);
    
    $curlHandles[$i] = $ch;
    curl_multi_add_handle($multiHandle, $ch);
}

// Processar resultados conforme ficam prontos
$running = null;
$totalComments = count($comments);

do {
    curl_multi_exec($multiHandle, $running);
    
    // Verificar se alguma requisição terminou
    while ($info = curl_multi_info_read($multiHandle)) {
        if ($info['msg'] == CURLMSG_DONE) {
            $ch = $info['handle'];
            $index = array_search($ch, $curlHandles);
            
            if ($index !== false) {
                $comment = $comments[$index];
                $resp = curl_multi_getcontent($ch);
                $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                
                if ($code === 200 && $resp) {
                    $data = json_decode($resp, true);
                    if (isset($data['attributeScores']) && !empty($data['attributeScores'])) {
                        $combined = [];
                        foreach ($data['attributeScores'] as $attr => $detail) {
                            $val = isset($detail['summaryScore']['value']) ? floatval($detail['summaryScore']['value']) : 0.0;
                            $combined[$attr] = $val;
                        }
                        $completed[$index] = analyze_one_fast($comment, $combined, $AI_THRESHOLDS);
                    } else {
                        $completed[$index] = ['color' => 'risk-blue', 'label' => 'Neutro', 'detail' => 'Análise indisponível'];
                    }
                } else {
                    $completed[$index] = ['color' => 'risk-blue', 'label' => 'Neutro', 'detail' => 'Análise indisponível'];
                }
                
                // Enviar resultado imediatamente via SSE
                sendSSE([
                    'index' => $index,
                    'result' => $completed[$index],
                    'processed' => count($completed),
                    'total' => $totalComments
                ]);
                
                curl_multi_remove_handle($multiHandle, $ch);
                curl_close($ch);
            }
        }
    }
    
    curl_multi_select($multiHandle);
} while ($running > 0);

curl_multi_close($multiHandle);

// Ordenar resultados pela ordem original
$results = [];
for ($i = 0; $i < $totalComments; $i++) {
    if (isset($completed[$i])) {
        $results[] = $completed[$i];
    }
}

sendSSE(['status' => 'complete', 'results' => $results]);
exit;
?>
