<?php
// Configurações do Sistema Bot Manager + Comentários
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Função para carregar variáveis do arquivo .env
function loadEnv($path = '.env') {
    if (!file_exists($path)) {
        return false;
    }
    
    $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos(trim($line), '#') === 0) {
            continue;
        }
        
        list($name, $value) = explode('=', $line, 2);
        $_ENV[trim($name)] = trim($value);
        putenv(trim($name) . '=' . trim($value));
    }
    return true;
}

// Carregar configurações do .env
loadEnv();

// Configurações do Banco de Dados - APENAS do .env
define('DB_HOST', $_ENV['DB_BOT_HOST']);
define('DB_USER', $_ENV['DB_BOT_USER']);
define('DB_PASS', $_ENV['DB_BOT_PASS']);
define('DB_NAME', $_ENV['DB_BOT_NAME']);

// Conexão com banco (com tratamento de erro)
try {
    $mysqli = @new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    
    if ($mysqli->connect_error) {
        // Se não conseguir conectar, tenta sem selecionar o banco
        $mysqli = @new mysqli(DB_HOST, DB_USER, DB_PASS);
        if ($mysqli->connect_error) {
            $mysqli = null;
        } else {
            $mysqli->set_charset("utf8mb4");
        }
    } else {
        $mysqli->set_charset("utf8mb4");
    }
} catch (Exception $e) {
    $mysqli = null;
}

// Configurações da Aplicação - APENAS do .env
define('APP_NAME', $_ENV['APP_NAME']);
define('APP_VERSION', $_ENV['APP_VERSION']);

// Função para verificar se o usuário está logado
function isLoggedIn() {
    return isset($_SESSION['user_logged']) && $_SESSION['user_logged'] === true;
}

// Função para fazer logout
function logout() {
    session_destroy();
    header('Location: login.php');
    exit;
}

// Função para redirecionar se não estiver logado
function requireLogin() {
    if (!isLoggedIn()) {
        header('Location: login.php');
        exit;
    }
}

// Configurações do Sistema de Comentários
// Conexão específica para comentários
$conn = new mysqli($_ENV['DB_COMMENTS_HOST'] ?? 'localhost', $_ENV['DB_COMMENTS_USER'] ?? 'root', $_ENV['DB_COMMENTS_PASS'] ?? '', $_ENV['DB_COMMENTS_NAME'] ?? 'database');

// Verificar conexão
if ($conn->connect_error) {
    die("Falha na conexão: " . $conn->connect_error);
}

// Definir caracteres UTF-8MB4 para suporte completo a emojis
$conn->set_charset("utf8mb4");

// Garantir que as tabelas tenham a coluna de aprovação
function ensure_approval_schema($conn, $tableName) {
    if (!($conn instanceof mysqli)) {
        return;
    }
    $tableNameSafe = preg_replace('/[^a-zA-Z0-9_]/', '', $tableName);
    $schema = 'su_co';

    // Primeiro verificar se a tabela existe
    $checkTableSql = "SELECT COUNT(*) AS cnt FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = '" . $conn->real_escape_string($schema) . "' AND TABLE_NAME = '" . $conn->real_escape_string($tableNameSafe) . "'";
    $resTable = $conn->query($checkTableSql);
    $tableExists = 0;
    if ($resTable) {
        $row = $resTable->fetch_assoc();
        $tableExists = intval($row['cnt'] ?? 0);
        $resTable->free();
    }
    
    // Se a tabela não existe, não fazer nada
    if ($tableExists === 0) {
        return;
    }

    // Verificar coluna via INFORMATION_SCHEMA
    $checkColSql = "SELECT COUNT(*) AS cnt FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = '" . $conn->real_escape_string($schema) . "' AND TABLE_NAME = '" . $conn->real_escape_string($tableNameSafe) . "' AND COLUMN_NAME = 'approved'";
    $resCol = $conn->query($checkColSql);
    $colExists = 0;
    if ($resCol) {
        $row = $resCol->fetch_assoc();
        $colExists = intval($row['cnt'] ?? 0);
        $resCol->free();
    }
    if ($colExists === 0) {
        // Tentar adicionar coluna; ignorar erro de coluna duplicada (1060)
        try {
            $conn->query("ALTER TABLE `{$tableNameSafe}` ADD COLUMN `approved` TINYINT(1) NOT NULL DEFAULT 0");
        } catch (mysqli_sql_exception $e) {
            if ((int)$e->getCode() !== 1060) { // 1060 = Duplicate column name
                throw $e;
            }
        }
    }

    // Verificar índice
    $checkIdxSql = "SHOW INDEX FROM `{$tableNameSafe}` WHERE Key_name = 'idx_approved'";
    $resIdx = $conn->query($checkIdxSql);
    $idxExists = ($resIdx && $resIdx->num_rows > 0);
    if ($resIdx) { $resIdx->free(); }
    if (!$idxExists) {
        try {
            $conn->query("ALTER TABLE `{$tableNameSafe}` ADD INDEX `idx_approved` (`approved`)");
        } catch (mysqli_sql_exception $e) {
            if ((int)$e->getCode() !== 1061) { // 1061 = Duplicate key name
                throw $e;
            }
        }
    }
}

// Aplicar para as tabelas relevantes
ensure_approval_schema($conn, 'comments');
ensure_approval_schema($conn, 'commentsaleatorio');
ensure_approval_schema($conn, 'Comments');

// Configurações da IA para análise de comentários - APENAS do .env
$AI_THRESHOLDS = [
    'TOXICITY' => floatval($_ENV['AI_TOXICITY_THRESHOLD']),
    'SEVERE_TOXICITY' => floatval($_ENV['AI_SEVERE_TOXICITY_THRESHOLD']),
    'INSULT' => floatval($_ENV['AI_INSULT_THRESHOLD']),
    'PROFANITY' => floatval($_ENV['AI_PROFANITY_THRESHOLD']),
    'THREAT' => floatval($_ENV['AI_THREAT_THRESHOLD']),
    'IDENTITY_ATTACK' => floatval($_ENV['AI_IDENTITY_ATTACK_THRESHOLD'])
];

$NEGATIVE_KEYWORDS = [
    'ruim', 'péssimo', 'horrível', 'terrível', 'merda', 'porcaria', 'lixo', 'fraco', 
    'decepcionado', 'insatisfeito', 'não recomendo', 'evitar', 'problema', 'defeito', 
    'quebrado', 'danificado', 'atraso', 'demora'
];

$POSITIVE_KEYWORDS = [
    'bom', 'ótimo', 'excelente', 'fantástico', 'maravilhoso', 'perfeito', 'recomendo', 
    'satisfeito', 'qualidade', 'superou', 'vale a pena', 'funciona', 'rápido', 'eficiente', 'prático'
];

$TIMEOUT_CONFIG = [
    'main_timeout' => intval($_ENV['MAIN_TIMEOUT']),
    'connect_timeout' => intval($_ENV['CONNECT_TIMEOUT']),
    'fallback_timeout' => intval($_ENV['FALLBACK_TIMEOUT']),
    'fallback_connect_timeout' => intval($_ENV['FALLBACK_CONNECT_TIMEOUT'])
];

// Carregar configurações salvas se existirem
$config_file = 'comentario/ia_config.json';
if (file_exists($config_file)) {
    $saved_config = json_decode(file_get_contents($config_file), true);
    if ($saved_config) {
        if (isset($saved_config['thresholds'])) {
            $AI_THRESHOLDS = array_merge($AI_THRESHOLDS, $saved_config['thresholds']);
        }
        if (isset($saved_config['negative_words'])) {
            $NEGATIVE_KEYWORDS = $saved_config['negative_words'];
        }
        if (isset($saved_config['positive_words'])) {
            $POSITIVE_KEYWORDS = $saved_config['positive_words'];
        }
        if (isset($saved_config['timeouts'])) {
            $TIMEOUT_CONFIG = array_merge($TIMEOUT_CONFIG, $saved_config['timeouts']);
        }
    }
}
?>