<?php
require_once 'config.php';

// Função auxiliar para exibir texto com emojis corretamente
function exibir_texto($texto) {
    return htmlspecialchars($texto, ENT_QUOTES, 'UTF-8');
}

// Configurar charset UTF-8 para suportar emojis
if (isset($conn) && $conn instanceof mysqli) {
    $conn->set_charset("utf8mb4");
}

// Verificar se é uma requisição POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método não permitido']);
    exit;
}

// Obter dados JSON
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data || !isset($data['comments']) || !is_array($data['comments'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Dados inválidos']);
    exit;
}

$comments = $data['comments'];
$results = [];

// Configurações da IA (já definidas no config.php)
$PERSPECTIVE_API_KEY_DEFAULT = "AIzaSyD_KQ77gkUJiGMGBqglxtg9vl5vwoJyTyM";
$PERSPECTIVE_API_KEY = getenv('PERSPECTIVE_API_KEY') ? getenv('PERSPECTIVE_API_KEY') : $PERSPECTIVE_API_KEY_DEFAULT;
$PERSPECTIVE_API_URL = "https://commentanalyzer.googleapis.com/v1alpha1/comments:analyze";
$PERSPECTIVE_LANGUAGES = array_map('trim', explode(',', getenv('PERSPECTIVE_LANGS') ? getenv('PERSPECTIVE_LANGS') : 'pt,en,es,fr'));
$ENABLE_AI_MODERATION = getenv('ENABLE_AI_MODERATION') ? (getenv('ENABLE_AI_MODERATION') === '1') : true;

// Função para chamar a API do Perspective
function perspective_call_language($text, $lang, $apiKey, $apiUrl) {
    global $TIMEOUT_CONFIG;
    $payload = [
        'comment' => ['text' => $text],
        'requestedAttributes' => [
            'TOXICITY' => new stdClass(),
            'SEVERE_TOXICITY' => new stdClass(),
            'INSULT' => new stdClass(),
            'PROFANITY' => new stdClass(),
            'THREAT' => new stdClass(),
            'IDENTITY_ATTACK' => new stdClass(),
        ],
        'doNotStore' => true,
        'languages' => [$lang],
    ];
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl . '?key=' . urlencode($apiKey));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_TIMEOUT, $TIMEOUT_CONFIG['main_timeout']);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $TIMEOUT_CONFIG['connect_timeout']);
    $resp = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($code === 200 && $resp) {
        $data = json_decode($resp, true);
        return isset($data['attributeScores']) ? $data['attributeScores'] : [];
    }
    return [];
}

// Função para analisar comentário com IA
function analyze_comment_ai($text, $languages, $apiKey, $apiUrl, $thresholds) {
    if (!$apiKey || empty($languages)) {
        return ['color' => 'risk-neutral', 'label' => 'IA off', 'maxScore' => 0.0, 'maxAttr' => ''];
    }
    $combined = [];
    foreach ($languages as $lang) {
        $scores = perspective_call_language($text, $lang, $apiKey, $apiUrl);
        foreach ($scores as $attr => $detail) {
            $val = isset($detail['summaryScore']['value']) ? floatval($detail['summaryScore']['value']) : 0.0;
            if (!isset($combined[$attr]) || $val > $combined[$attr]) {
                $combined[$attr] = $val;
            }
        }
    }
    if (empty($combined)) {
        return ['color' => 'risk-neutral', 'label' => 'Sem score', 'maxScore' => 0.0, 'maxAttr' => ''];
    }
    // Encontra atributo com maior score
    $maxAttr = '';
    $maxScore = -1.0;
    foreach ($combined as $attr => $val) {
        if ($val > $maxScore) { $maxScore = $val; $maxAttr = $attr; }
    }
    // Classificação por cores (verde, amarelo, laranja, vermelho)
    $color = 'risk-green';
    $hit = false;
    foreach ($thresholds as $attr => $th) {
        $val = isset($combined[$attr]) ? $combined[$attr] : 0.0;
        if ($val >= $th) { $color = 'risk-red'; $hit = true; break; }
    }
    if (!$hit) {
        foreach ($thresholds as $attr => $th) {
            $val = isset($combined[$attr]) ? $combined[$attr] : 0.0;
            if ($val >= $th * 0.75) { $color = 'risk-orange'; $hit = true; break; }
        }
    }
    if (!$hit) {
        foreach ($thresholds as $attr => $th) {
            $val = isset($combined[$attr]) ? $combined[$attr] : 0.0;
            if ($val >= $th * 0.5) { $color = 'risk-yellow'; $hit = true; break; }
        }
    }
    $label = $maxAttr !== '' ? ($maxAttr . ': ' . number_format($maxScore, 2)) : 'OK';
    return ['color' => $color, 'label' => $label, 'maxScore' => $maxScore, 'maxAttr' => $maxAttr];
}

// Processar cada comentário
foreach ($comments as $comment) {
    if (empty($comment)) {
        $results[] = ['color' => 'risk-neutral', 'label' => 'Vazio', 'maxScore' => 0.0, 'maxAttr' => ''];
        continue;
    }
    
    // Análise com IA
    $ai_result = analyze_comment_ai($comment, $PERSPECTIVE_LANGUAGES, $PERSPECTIVE_API_KEY, $PERSPECTIVE_API_URL, $AI_THRESHOLDS);
    
    // Análise com palavras-chave (fallback)
    $keyword_result = ['color' => 'risk-neutral', 'label' => 'OK', 'maxScore' => 0.0, 'maxAttr' => ''];
    
    $comment_lower = strtolower($comment);
    $negative_count = 0;
    $positive_count = 0;
    
    foreach ($NEGATIVE_KEYWORDS as $keyword) {
        if (strpos($comment_lower, strtolower($keyword)) !== false) {
            $negative_count++;
        }
    }
    
    foreach ($POSITIVE_KEYWORDS as $keyword) {
        if (strpos($comment_lower, strtolower($keyword)) !== false) {
            $positive_count++;
        }
    }
    
    if ($negative_count > $positive_count) {
        $keyword_result = ['color' => 'risk-yellow', 'label' => 'Negativo (' . $negative_count . ')', 'maxScore' => 0.5, 'maxAttr' => 'NEGATIVE'];
    } elseif ($positive_count > $negative_count) {
        $keyword_result = ['color' => 'risk-green', 'label' => 'Positivo (' . $positive_count . ')', 'maxScore' => 0.2, 'maxAttr' => 'POSITIVE'];
    }
    
    // Usar resultado da IA se disponível, senão usar palavras-chave
    if ($ai_result['maxScore'] > 0) {
        $results[] = $ai_result;
    } else {
        $results[] = $keyword_result;
    }
}

// Retornar resultados em JSON
header('Content-Type: application/json');
echo json_encode(['results' => $results]);
?>
