<?php
require_once 'config.php';
require_once 'config_ia.php';
require_once 'functions.php';

// -------- IA (Perspective) para análise de risco do comentário --------
// Config (usa ENV se existir, senão default)
$PERSPECTIVE_API_KEY_DEFAULT = "AIzaSyD_KQ77gkUJiGMGBqglxtg9vl5vwoJyTyM";
$PERSPECTIVE_API_KEY = getenv('PERSPECTIVE_API_KEY') ? getenv('PERSPECTIVE_API_KEY') : $PERSPECTIVE_API_KEY_DEFAULT;
$PERSPECTIVE_API_URL = "https://commentanalyzer.googleapis.com/v1alpha1/comments:analyze";
$PERSPECTIVE_LANGUAGES = array_map('trim', explode(',', getenv('PERSPECTIVE_LANGS') ? getenv('PERSPECTIVE_LANGS') : 'pt,en,es,fr'));
$ENABLE_AI_MODERATION = getenv('ENABLE_AI_MODERATION') ? (getenv('ENABLE_AI_MODERATION') === '1') : true;

function perspective_call_language($text, $lang, $apiKey, $apiUrl) {
    global $TIMEOUT_CONFIG;
    $payload = [
        'comment' => ['text' => $text],
        'requestedAttributes' => [
            'TOXICITY' => new stdClass(),
            'SEVERE_TOXICITY' => new stdClass(),
            'INSULT' => new stdClass(),
            'PROFANITY' => new stdClass(),
            'THREAT' => new stdClass(),
            'IDENTITY_ATTACK' => new stdClass(),
        ],
        'doNotStore' => true,
        'languages' => [$lang],
    ];
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $apiUrl . '?key=' . urlencode($apiKey));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_TIMEOUT, $TIMEOUT_CONFIG['main_timeout']);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $TIMEOUT_CONFIG['connect_timeout']);
    $resp = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($code === 200 && $resp) {
        $data = json_decode($resp, true);
        return isset($data['attributeScores']) ? $data['attributeScores'] : [];
    }
    return [];
}

function analyze_comment_ai($text, $languages, $apiKey, $apiUrl, $thresholds) {
    if (!$apiKey || empty($languages)) {
        return ['color' => 'risk-neutral', 'label' => 'IA off', 'maxScore' => 0.0, 'maxAttr' => ''];
    }
    $combined = [];
    foreach ($languages as $lang) {
        $scores = perspective_call_language($text, $lang, $apiKey, $apiUrl);
        foreach ($scores as $attr => $detail) {
            $val = isset($detail['summaryScore']['value']) ? floatval($detail['summaryScore']['value']) : 0.0;
            if (!isset($combined[$attr]) || $val > $combined[$attr]) {
                $combined[$attr] = $val;
            }
        }
    }
    if (empty($combined)) {
        return ['color' => 'risk-neutral', 'label' => 'Sem score', 'maxScore' => 0.0, 'maxAttr' => ''];
    }
    // Encontra atributo com maior score
    $maxAttr = '';
    $maxScore = -1.0;
    foreach ($combined as $attr => $val) {
        if ($val > $maxScore) { $maxScore = $val; $maxAttr = $attr; }
    }
    // Classificação por cores (verde, amarelo, laranja, vermelho)
    $color = 'risk-green';
    $hit = false;
    foreach ($thresholds as $attr => $th) {
        $val = isset($combined[$attr]) ? $combined[$attr] : 0.0;
        if ($val >= $th) { $color = 'risk-red'; $hit = true; break; }
    }
    if (!$hit) {
        foreach ($thresholds as $attr => $th) {
            $val = isset($combined[$attr]) ? $combined[$attr] : 0.0;
            if ($val >= $th * 0.75) { $color = 'risk-orange'; $hit = true; break; }
        }
    }
    if (!$hit) {
        foreach ($thresholds as $attr => $th) {
            $val = isset($combined[$attr]) ? $combined[$attr] : 0.0;
            if ($val >= $th * 0.5) { $color = 'risk-yellow'; $hit = true; break; }
        }
    }
    $label = $maxAttr !== '' ? ($maxAttr . ': ' . number_format($maxScore, 2)) : 'OK';
    return ['color' => $color, 'label' => $label, 'maxScore' => $maxScore, 'maxAttr' => $maxAttr];
}

// Lidar com exclusões
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id'], $_POST['source'])) {
    $delete_id = intval($_POST['delete_id']);
    $source = $_POST['source'];

    if (in_array($source, ['comments', 'commentsaleatorio'])) {
        $delete_sql = "DELETE FROM $source WHERE id = ?";
        $stmt = $conn->prepare($delete_sql);
        if ($stmt) {
            $stmt->bind_param('i', $delete_id);
            $stmt->execute();
            $stmt->close();
        }
    }
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Lidar com exclusões de todos com mesmo link
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_link'], $_POST['source'])) {
    $delete_link = $_POST['delete_link'];
    $source = $_POST['source'];

    if (in_array($source, ['comments', 'commentsaleatorio'])) {
        $delete_sql = "DELETE FROM $source WHERE link = ?";
        $stmt = $conn->prepare($delete_sql);
        if ($stmt) {
            $stmt->bind_param('s', $delete_link);
            $stmt->execute();
            $stmt->close();
        }
    }
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Lidar com exclusão em massa (selecionados)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'bulk_delete') {
    $ids = isset($_POST['ids']) && is_array($_POST['ids']) ? array_map('intval', $_POST['ids']) : [];
    $sources = isset($_POST['sources']) && is_array($_POST['sources']) ? $_POST['sources'] : [];
    foreach ($ids as $oneId) {
        $tbl = isset($sources[$oneId]) ? preg_replace('/[^a-zA-Z0-9_]/', '', $sources[$oneId]) : '';
        if (!in_array($tbl, ['comments', 'commentsaleatorio'])) { continue; }
        $stmt = $conn->prepare("DELETE FROM `$tbl` WHERE id = ?");
        if ($stmt) { $stmt->bind_param('i', $oneId); $stmt->execute(); $stmt->close(); }
    }
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Filtro de aprovação: mostrar apenas não aprovados
$approved_condition = "approved = 0";

// Configurar filtro de gênero
$filter_gender = isset($_GET['gender']) ? intval($_GET['gender']) : 0; // 0 = todos, 1 = masculino, 2 = feminino, 3 = unissex

// Configurar paginação
$limit = 50;
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$offset = ($page - 1) * $limit;

// Construir condições por tabela (com approved = 0)
$where_comments = "WHERE $approved_condition";
if ($filter_gender > 0) {
    if ($filter_gender == 4) {
        $where_comments = "WHERE gender IS NULL AND $approved_condition";
    } else {
        $where_comments = "WHERE gender = $filter_gender AND $approved_condition";
    }
}

$where_random = "WHERE $approved_condition";
if ($filter_gender > 0) {
    if ($filter_gender == 4) {
        // A tabela commentsaleatorio não possui gender NULL nessa visão de pendentes
        $where_random = "WHERE 1=0";
    } else {
        $where_random = "WHERE gender = $filter_gender AND $approved_condition";
    }
}

// Consultar dados combinados para pendentes
$sql_pendente = "
    SELECT id, order_id, link, comment, gender, 'comments' AS source, 1 AS priority
    FROM comments
    $where_comments
    UNION ALL
    SELECT id, order_id, link, comment, gender, 'commentsaleatorio' AS source, 2 AS priority
    FROM commentsaleatorio
    $where_random
    ORDER BY priority ASC, id ASC
    LIMIT $limit OFFSET $offset
";
$result_pendente = $conn->query($sql_pendente);

// Contar total de registros pendentes por tipo de fonte (apenas não aprovados)
if ($filter_gender == 4) {
    $total_personalizado = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender IS NULL AND $approved_condition")->fetch_assoc()['total'];
    $total_aleatorio = 0; // A tabela commentsaleatorio não tem registros sem gênero (gender é NULL)
} else {
    $total_personalizado = $conn->query("SELECT COUNT(*) AS total FROM comments " . ($where_comments ? $where_comments : ""))->fetch_assoc()['total'];
    $total_aleatorio = $conn->query("SELECT COUNT(*) AS total FROM commentsaleatorio " . ($where_random ? $where_random : ""))->fetch_assoc()['total'];
}
$total_pendentes = $total_personalizado + $total_aleatorio;

// Contar totais por gênero (apenas não aprovados)
$total_masculino = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender = 1 AND $approved_condition")->fetch_assoc()['total'] +
                  $conn->query("SELECT COUNT(*) AS total FROM commentsaleatorio WHERE gender = 1 AND $approved_condition")->fetch_assoc()['total'];
$total_feminino = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender = 2 AND $approved_condition")->fetch_assoc()['total'] +
                 $conn->query("SELECT COUNT(*) AS total FROM commentsaleatorio WHERE gender = 2 AND $approved_condition")->fetch_assoc()['total'];
$total_unissex = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender = 3 AND $approved_condition")->fetch_assoc()['total'] +
                $conn->query("SELECT COUNT(*) AS total FROM commentsaleatorio WHERE gender = 3 AND $approved_condition")->fetch_assoc()['total'];

// Calcular total de comentários sem gênero definido (apenas não aprovados)
$total_indefinido = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender IS NULL AND $approved_condition")->fetch_assoc()['total'];

// Função auxiliar para construir URL de paginação
function buildPaginationUrl($page_num) {
    $params = [];
    
    foreach ($_GET as $key => $value) {
        if ($key !== 'page') {
            $params[$key] = $value;
        }
    }
    
    $params['page'] = $page_num;
    
    return '?' . http_build_query($params);
}

// Função para criar links de filtro por gênero
function buildGenderFilterUrl($gender_id) {
    $params = [];
    
    foreach ($_GET as $key => $value) {
        if ($key !== 'gender' && $key !== 'page') {
            $params[$key] = $value;
        }
    }
    
    if ($gender_id > 0) {
        $params['gender'] = $gender_id;
    }
    
    $query_string = http_build_query($params);
    return empty($query_string) ? '?' : '?' . $query_string;
}

// Função para exibir nome do gênero
function getNomeGenero($gender_id) {
    if ($gender_id === null) {
        return "Não definido";
    }
    
    switch ($gender_id) {
        case 1:
            return "Masculino";
        case 2:
            return "Feminino";
        case 3:
            return "Unissex";
        default:
            return "Não definido";
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
    <title>Comentários</title>
    <link rel="icon" href="icon.png" type="image/png">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="popup_config.css">
    <style>
        
        .totals {
            display: flex;
            gap: 1rem;
            margin-bottom: 1rem;
        }
        
        .totals span {
            padding: 0.5rem 1rem;
            background-color: var(--color-surface-variant);
            border-radius: var(--border-radius-md);
            font-size: var(--font-size-sm);
            color: var(--color-text-secondary);
        }
        
        .total-link {
            text-decoration: none;
            transition: all 0.2s ease;
        }
        
        .total-link span {
            padding: 0.5rem 1rem;
            background-color: var(--color-surface-variant);
            border-radius: var(--border-radius-md);
            font-size: var(--font-size-sm);
            color: var(--color-text-secondary);
            transition: all 0.2s ease;
        }
        
        .total-link.active span {
            background-color: var(--color-primary-light);
            color: var(--color-primary);
            font-weight: 600;
        }
        
        .total-link:hover span {
            background-color: var(--color-primary-light);
            color: var(--color-primary);
        }
        
        .main-menu {
            display: flex;
            background-color: var(--color-surface);
            border-radius: var(--border-radius-lg);
            box-shadow: var(--shadow-md);
            overflow: hidden;
            margin-bottom: 2rem;
        }
        
        .main-menu a {
            padding: 1rem 1.5rem;
            color: var(--color-text-secondary);
            text-decoration: none;
            font-weight: 600;
            display: flex;
            align-items: center;
            transition: all var(--transition-base);
            border-bottom: 3px solid transparent;
        }
        
        .main-menu a.active {
            color: var(--color-primary);
            background-color: var(--color-primary-light);
            border-bottom-color: var(--color-primary);
        }
        
        .main-menu a:hover:not(.active) {
            background-color: var(--color-surface-variant);
            color: var(--color-text-primary);
        }
        
        .main-menu i {
            margin-right: 0.5rem;
        }

        @media (min-width: 1200px) {
            .table td.col-link, 
            .table td.col-comment {
                padding-right: 10px;
            }
            
            .table .truncate {
                width: 100%;
                display: block;
            }
        }
        
        .col-gender {
            width: 100px;
            text-align: center;
        }
        
        td.col-gender {
            font-size: 0.9em;
        }
        
        .action-icon.delete-all {
            background-color: #dc3545;
            margin-left: 5px;
        }
        
        .action-icon.delete-all:hover {
            background-color: #bd2130;
        }
    </style>
    <script>
        document.addEventListener('DOMContentLoaded', () => {
            const copyLinkCells = document.querySelectorAll('td.col-link');
            const copyCommentCells = document.querySelectorAll('td.col-comment');

            copyLinkCells.forEach(cell => {
                cell.addEventListener('click', () => {
                    const originalLink = cell.getAttribute('title') || cell.textContent.trim();
                    navigator.clipboard.writeText(originalLink).then(() => {
                        cell.classList.add('copied');
                        setTimeout(() => {
                            cell.classList.remove('copied');
                        }, 1000);
                    }).catch(err => {
                        console.error('Erro ao copiar o link: ', err);
                    });
                });
            });

            copyCommentCells.forEach(cell => {
                cell.addEventListener('click', () => {
                    const commentText = cell.getAttribute('title') || cell.textContent.trim();
                    navigator.clipboard.writeText(commentText).then(() => {
                        cell.classList.add('copied');
                        setTimeout(() => {
                            cell.classList.remove('copied');
                        }, 1000);
                    }).catch(err => {
                        console.error('Erro ao copiar o comentário: ', err);
                    });
                });
            });
        });
    </script>
</head>
<body>
    <!-- Navbar -->
    <?php include 'navbar.php'; ?>

    <main class="page-container">
        <?php
        // Consultar métricas gerais
        // Total de ações hoje
        $current_date = date('Y-m-d');
        $total_acoes_hoje = $conn->query("
            SELECT COALESCE(SUM(actions_today), 0) AS total 
            FROM (
                SELECT SUM(actions_today) AS actions_today FROM comentario01 WHERE DATE(last_action_date) = '$current_date'
                UNION ALL
                SELECT SUM(actions_today) AS actions_today FROM comentario02 WHERE DATE(last_action_date) = '$current_date'
                UNION ALL
                SELECT SUM(actions_today) AS actions_today FROM comentario03 WHERE DATE(last_action_date) = '$current_date'
            ) AS totals
        ")->fetch_assoc()['total'] ?? 0;
        
        // Total de ações ontem
        $yesterday_date = date('Y-m-d', strtotime('-1 day'));
        $acoes_ontem = $conn->query("
            SELECT COUNT(*) AS total 
            FROM history 
            WHERE DATE(data_acao) = '$yesterday_date'
        ")->fetch_assoc()['total'] ?? 0;
        
        // Ações na última hora (60 minutos exatos)
        $acoes_ultima_hora = $conn->query("
            SELECT COUNT(*) AS total 
            FROM history 
            WHERE data_acao > DATE_SUB(NOW(), INTERVAL 60 MINUTE)
        ")->fetch_assoc()['total'] ?? 0;
        
        // Data da última ação
        $ultima_acao = $conn->query("
            SELECT MAX(last_action_date) AS last_date FROM (
                SELECT MAX(last_action_date) AS last_action_date FROM comentario01
                UNION ALL
                SELECT MAX(last_action_date) AS last_action_date FROM comentario02
                UNION ALL
                SELECT MAX(last_action_date) AS last_action_date FROM comentario03
            ) AS dates
        ")->fetch_assoc()['last_date'] ?? 'N/A';
        
        // Formatar data da última ação
        $ultima_acao_formatada = $ultima_acao != 'N/A' ? date('d/m/Y H:i:s', strtotime($ultima_acao)) : 'N/A';
        
        // Totais globais (pendentes para aprovar e aprovados) para o cabeçalho
        $total_pendentes_aprovar_header = $conn->query("
            SELECT 
                (SELECT COUNT(*) FROM comments WHERE approved = 0) + 
                (SELECT COUNT(*) FROM commentsaleatorio WHERE approved = 0) AS total
        ")->fetch_assoc()['total'] ?? 0;
        $total_aprovados_header = $conn->query("
            SELECT 
                (SELECT COUNT(*) FROM comments WHERE approved = 1) + 
                (SELECT COUNT(*) FROM commentsaleatorio WHERE approved = 1) AS total
        ")->fetch_assoc()['total'] ?? 0;

        // Calcular previsão para zerar (baseada nos aprovados prontos)
        $previsao_zerar = 'N/A';
        if ($acoes_ultima_hora > 0 && $total_aprovados_header > 0) {
            $minutos_totais = ceil(($total_aprovados_header / $acoes_ultima_hora) * 60);
            if ($minutos_totais < 60) {
                $previsao_zerar = $minutos_totais . ' min';
            } else {
                $horas = floor($minutos_totais / 60);
                $minutos = $minutos_totais % 60;
                if ($horas < 24) {
                    $previsao_zerar = $horas . 'h ' . $minutos . 'min';
                } else {
                    $dias = floor($horas / 24);
                    $horas_restantes = $horas % 24;
                    $previsao_zerar = $dias . ' dias, ' . $horas_restantes . 'h ' . $minutos . 'min';
                }
            }
        }
        ?>
        
        <div class="metrics-summary">
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo number_format($total_acoes_hoje, 0, ',', '.'); ?></div>
                    <div class="metric-label">Ações Hoje <span style="font-size: 0.8em; opacity: 0.7;">(Ontem: <?php echo number_format($acoes_ontem, 0, ',', '.'); ?>)</span></div>
                </div>
            </div>
            
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-hourglass-half"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo number_format($total_pendentes_aprovar_header, 0, ',', '.'); ?></div>
                    <div class="metric-label">Pendentes p/ aprovar</div>
                </div>
            </div>
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-check-circle"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo number_format($total_aprovados_header, 0, ',', '.'); ?></div>
                    <div class="metric-label">Aprovados (prontos)</div>
                </div>
            </div>
            
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-tachometer-alt"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo number_format($acoes_ultima_hora, 0, ',', '.'); ?></div>
                    <div class="metric-label">Ações na última hora</div>
                </div>
            </div>
            
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-hourglass-end"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo $previsao_zerar; ?></div>
                    <div class="metric-label">Previsão para zerar</div>
                </div>
            </div>
            
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-calendar-check"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo $ultima_acao_formatada; ?></div>
                    <div class="metric-label">Última Ação</div>
                </div>
            </div>
        </div>
        

        <div class="totals">
            <span>Total Personalizado: <?php echo $total_personalizado; ?></span>
            <span>Total Aleatório: <?php echo $total_aleatorio; ?></span>
            <a href="<?php echo buildGenderFilterUrl(0); ?>" class="total-link <?php echo $filter_gender == 0 ? 'active' : ''; ?>">
                <span>Total Geral: <?php echo $total_pendentes; ?></span>
            </a>
            <a href="<?php echo buildGenderFilterUrl(1); ?>" class="total-link <?php echo $filter_gender == 1 ? 'active' : ''; ?>">
                <span>Total Masculino: <?php echo $total_masculino; ?></span>
            </a>
            <a href="<?php echo buildGenderFilterUrl(2); ?>" class="total-link <?php echo $filter_gender == 2 ? 'active' : ''; ?>">
                <span>Total Feminino: <?php echo $total_feminino; ?></span>
            </a>
            <a href="<?php echo buildGenderFilterUrl(3); ?>" class="total-link <?php echo $filter_gender == 3 ? 'active' : ''; ?>">
                <span>Total Unissex: <?php echo $total_unissex; ?></span>
            </a>
            <?php if ($total_indefinido > 0): ?>
            <a href="<?php echo buildGenderFilterUrl(4); ?>" class="total-link <?php echo $filter_gender == 4 ? 'active' : ''; ?>">
                <span>Não definido: <?php echo $total_indefinido; ?></span>
            </a>
            <?php endif; ?>
        </div>

        <div class="card">
          
            <div class="card-body">
                <?php if (isset($result_pendente) && $result_pendente->num_rows > 0): ?>
                    <div class="table-responsive">
                        <div class="bulk-actions" style="margin-bottom:10px; display:flex; gap:8px; align-items:center;">
                            <button type="button" id="bulkApprove" class="btn btn-primary btn-sm">
                                <i class="fas fa-check"></i>Aprovar Selecionados
                            </button>
                            <button type="button" id="bulkDelete" class="btn btn-secondary btn-sm">
                                <i class="fas fa-trash-alt"></i>Excluir Selecionados
                            </button>
                        </div>
                        <table class="table">
                            <thead>
                                <tr>
                                    <th style="width:50px;">
                                        <input type="checkbox" id="selectAll">
                                    </th>
                                    <th class="col-order-id">Order ID</th>
                                    <th class="col-link">Link</th>
                                    <th class="col-comment">Comentário</th>
                                    <th style="width:140px;">IA</th>
                                    <th class="col-gender">Gênero</th>
                                    <th class="col-actions">Ações</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($row = $result_pendente->fetch_assoc()): ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" class="row-check" value="<?php echo $row['id']; ?>" data-source="<?php echo exibir_texto($row['source']); ?>">
                                        </td>
                                        <td><?php echo exibir_texto($row['order_id']); ?></td>
                                        <td class="col-link" title="<?php echo exibir_texto($row['link']); ?>">
                                            <span class="truncate"><?php echo exibir_texto($row['link']); ?></span>
                                        </td>
                                        <td class="col-comment" title="<?php echo exibir_texto($row['comment']); ?>">
                                            <span class="truncate"><?php echo exibir_texto($row['comment']); ?></span>
                                        </td>
                                        <td>
                                            <div class="ai-cell">
                                                <span class="ai-dot risk-neutral"></span>
                                                <span class="ai-badge risk-neutral ai-pending" data-comment="<?php echo htmlspecialchars($row['comment']); ?>">Analisando...</span>
                                            </div>
                                        </td>
                                        <td class="col-gender"><?php echo getNomeGenero($row['gender']); ?></td>
                                        <td class="col-actions">
                                            <form method="POST" action="approve.php" style="margin: 0; display: inline-block;">
                                                <input type="hidden" name="id" value="<?php echo $row['id']; ?>">
                                                <input type="hidden" name="source" value="<?php echo exibir_texto($row['source']); ?>">
                                                <input type="hidden" name="action" value="approve">
                                                <button type="submit" class="action-icon view" title="Aprovar">
                                                    <i class="fas fa-check"></i>
                                                </button>
                                            </form>
                                            <form method="POST" action="approve.php" style="margin: 0; display: inline-block;">
                                                <input type="hidden" name="link" value="<?php echo exibir_texto($row['link']); ?>">
                                                <input type="hidden" name="source" value="<?php echo exibir_texto($row['source']); ?>">
                                                <input type="hidden" name="action" value="approve_link">
                                                <button type="submit" class="action-icon view" title="Aprovar todos com mesmo link">
                                                    <i class="fas fa-check-double"></i>
                                                </button>
                                            </form>
                                            
                                            <form method="POST" style="margin: 0; display: inline-block;">
                                                <input type="hidden" name="delete_id" value="<?php echo $row['id']; ?>">
                                                <input type="hidden" name="source" value="<?php echo exibir_texto($row['source']); ?>">
                                                <button type="submit" class="action-icon delete" title="Excluir">
                                                    <i class="fas fa-trash-alt"></i>
                                                </button>
                                            </form>
                                            <form method="POST" style="margin: 0; display: inline-block;">
                                                <input type="hidden" name="delete_link" value="<?php echo exibir_texto($row['link']); ?>">
                                                <input type="hidden" name="source" value="<?php echo exibir_texto($row['source']); ?>">
                                                <button type="submit" class="action-icon delete-all" title="Excluir todos com mesmo link" onclick="return confirm('Tem certeza que deseja excluir todos os comentários com este mesmo link?');">
                                                    <i class="fas fa-dumpster"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                    <script>
                        (function(){
                            const selectAll = document.getElementById('selectAll');
                            const rowChecks = () => Array.from(document.querySelectorAll('.row-check'));
                            const setAll = (checked) => rowChecks().forEach(cb => { cb.checked = checked; });
                            const syncHeader = () => {
                                const cbs = rowChecks();
                                const all = cbs.length > 0 && cbs.every(cb => cb.checked);
                                const any = cbs.some(cb => cb.checked);
                                if (selectAll) {
                                    selectAll.checked = all;
                                    selectAll.indeterminate = !all && any;
                                }
                            };
                            if (selectAll) selectAll.addEventListener('change', e => { setAll(e.target.checked); syncHeader(); });
                            document.addEventListener('change', e => { if (e.target.classList && e.target.classList.contains('row-check')) syncHeader(); });

                            function submitBulk(action){
                                const selected = rowChecks().filter(cb => cb.checked);
                                if (selected.length === 0) { alert('Selecione pelo menos um item.'); return; }
                                const form = document.createElement('form');
                                form.method = 'POST';
                                form.action = 'approve.php';
                                form.style.display = 'none';
                                const inpAction = document.createElement('input');
                                inpAction.type = 'hidden'; inpAction.name = 'action'; inpAction.value = 'bulk';
                                form.appendChild(inpAction);
                                const inpBulk = document.createElement('input');
                                inpBulk.type = 'hidden'; inpBulk.name = 'bulk_action'; inpBulk.value = action;
                                form.appendChild(inpBulk);
                                selected.forEach(cb => {
                                    const idInput = document.createElement('input');
                                    idInput.type = 'hidden'; idInput.name = 'ids[]'; idInput.value = cb.value;
                                    form.appendChild(idInput);
                                    const srcInput = document.createElement('input');
                                    srcInput.type = 'hidden'; srcInput.name = `sources[${cb.value}]`; srcInput.value = cb.getAttribute('data-source');
                                    form.appendChild(srcInput);
                                });
                                document.body.appendChild(form);
                                form.submit();
                            }
                            function submitBulkDelete(){
                                const selected = rowChecks().filter(cb => cb.checked);
                                if (selected.length === 0) { alert('Selecione pelo menos um item.'); return; }
                                const form = document.createElement('form');
                                form.method = 'POST';
                                form.action = '';
                                form.style.display = 'none';
                                const inpAction = document.createElement('input');
                                inpAction.type = 'hidden'; inpAction.name = 'action'; inpAction.value = 'bulk_delete';
                                form.appendChild(inpAction);
                                selected.forEach(cb => {
                                    const idInput = document.createElement('input');
                                    idInput.type = 'hidden'; idInput.name = 'ids[]'; idInput.value = cb.value;
                                    form.appendChild(idInput);
                                    const srcInput = document.createElement('input');
                                    srcInput.type = 'hidden'; srcInput.name = `sources[${cb.value}]`; srcInput.value = cb.getAttribute('data-source');
                                    form.appendChild(srcInput);
                                });
                                document.body.appendChild(form);
                                form.submit();
                            }
                            const bulkApprove = document.getElementById('bulkApprove');
                            if (bulkApprove) bulkApprove.addEventListener('click', () => submitBulk('approve'));
                            const bulkDelete = document.getElementById('bulkDelete');
                            if (bulkDelete) bulkDelete.addEventListener('click', submitBulkDelete);
                        })();
                    </script>
                    <div class="pagination">
                        <?php for ($i = 1; $i <= ceil($total_pendentes / $limit); $i++): ?>
                            <a href="<?php echo buildPaginationUrl($i); ?>" class="<?php echo $page == $i ? 'active' : ''; ?>">
                                <?php echo $i; ?>
                            </a>
                        <?php endfor; ?>
                    </div>
                <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-check-circle"></i>
                        <p>Não há comentários pendentes</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <!-- Popup de Configuração da IA -->
    <div id="configPopup" class="config-popup">
        <div class="config-popup-content">
            <div class="config-popup-header">
                <h2><i class="fas fa-robot"></i> Configuração da IA</h2>
                <button class="config-popup-close" onclick="closeConfigPopup()">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="config-popup-body">
                <div id="configAlert"></div>
                
                <div class="config-section">
                    <h3><i class="fas fa-sliders-h"></i> Thresholds da IA</h3>
                    <p class="config-config-help-text">Valores entre 0.0 e 1.0. Valores menores = mais sensível (detecta mais problemas)</p>
                    
                    <div class="thresholds-grid">
                        <div class="threshold-item">
                            <label for="threshold_toxicity">Critica</label>
                            <input type="number" id="threshold_toxicity" min="0" max="1" step="0.01" value="<?php echo $AI_THRESHOLDS['TOXICITY']; ?>">
                            <div class="config-config-help-text">TOXICITY</div>
                        </div>
                        <div class="threshold-item">
                            <label for="threshold_severe">Critica grave</label>
                            <input type="number" id="threshold_severe" min="0" max="1" step="0.01" value="<?php echo $AI_THRESHOLDS['SEVERE_TOXICITY']; ?>">
                            <div class="config-config-help-text">SEVERE_TOXICITY</div>
                        </div>
                        <div class="threshold-item">
                            <label for="threshold_insult">Insulto</label>
                            <input type="number" id="threshold_insult" min="0" max="1" step="0.01" value="<?php echo $AI_THRESHOLDS['INSULT']; ?>">
                            <div class="config-help-text">INSULT</div>
                        </div>
                        <div class="threshold-item">
                            <label for="threshold_profanity">Linguagem imprópria</label>
                            <input type="number" id="threshold_profanity" min="0" max="1" step="0.01" value="<?php echo $AI_THRESHOLDS['PROFANITY']; ?>">
                            <div class="config-help-text">PROFANITY</div>
                        </div>
                        <div class="threshold-item">
                            <label for="threshold_threat">Ameaça</label>
                            <input type="number" id="threshold_threat" min="0" max="1" step="0.01" value="<?php echo $AI_THRESHOLDS['THREAT']; ?>">
                            <div class="config-help-text">THREAT</div>
                        </div>
                        <div class="threshold-item">
                            <label for="threshold_identity">Ataque</label>
                            <input type="number" id="threshold_identity" min="0" max="1" step="0.01" value="<?php echo $AI_THRESHOLDS['IDENTITY_ATTACK']; ?>">
                            <div class="config-help-text">IDENTITY_ATTACK</div>
                        </div>
                    </div>
                </div>
                
                <div class="config-section">
                    <h3><i class="fas fa-keyboard"></i> Palavras-Chave para Análise</h3>
                    <p class="config-help-text">Usadas quando a API da IA falha. Separe por vírgulas.</p>
                    
                    <div class="words-section">
                        <div class="words-item">
                            <h4><i class="fas fa-thumbs-down"></i> Palavras Negativas</h4>
                            <textarea id="negative_words" placeholder="ruim, péssimo, horrível, merda, problema..."><?php echo implode(', ', $NEGATIVE_KEYWORDS); ?></textarea>
                            <div class="config-help-text">Palavras que indicam comentários negativos</div>
                        </div>
                        
                        <div class="words-item">
                            <h4><i class="fas fa-thumbs-up"></i> Palavras Positivas</h4>
                            <textarea id="positive_words" placeholder="bom, ótimo, excelente, recomendo, qualidade..."><?php echo implode(', ', $POSITIVE_KEYWORDS); ?></textarea>
                            <div class="config-help-text">Palavras que indicam comentários positivos</div>
                        </div>
                    </div>
                </div>
                
                <div class="config-section">
                    <h3><i class="fas fa-clock"></i> Configurações de Timeout</h3>
                    <p class="config-help-text">Tempos em segundos para requisições da API</p>
                    
                    <div class="timeouts-grid">
                        <div class="threshold-item">
                            <label for="timeout_main">Timeout Principal</label>
                            <input type="number" id="timeout_main" min="1" max="300" value="<?php echo $TIMEOUT_CONFIG['main_timeout']; ?>">
                            <div class="config-help-text">Tempo limite para análise principal</div>
                        </div>
                        
                        <div class="threshold-item">
                            <label for="timeout_connect">Timeout Conexão</label>
                            <input type="number" id="timeout_connect" min="1" max="120" value="<?php echo $TIMEOUT_CONFIG['connect_timeout']; ?>">
                            <div class="config-help-text">Tempo limite para conectar</div>
                        </div>
                        
                        <div class="threshold-item">
                            <label for="timeout_fallback">Timeout Fallback</label>
                            <input type="number" id="timeout_fallback" min="1" max="300" value="<?php echo $TIMEOUT_CONFIG['fallback_timeout']; ?>">
                            <div class="config-help-text">Tempo limite para análise de fallback</div>
                        </div>
                        
                        <div class="threshold-item">
                            <label for="timeout_fallback_connect">Timeout Fallback Conexão</label>
                            <input type="number" id="timeout_fallback_connect" min="1" max="120" value="<?php echo $TIMEOUT_CONFIG['fallback_connect_timeout']; ?>">
                            <div class="config-help-text">Tempo limite para conectar no fallback</div>
                        </div>
                    </div>
                </div>
                
                
                <div class="config-buttons">
                    <button type="button" class="config-btn config-btn-success" onclick="saveConfig()">
                        <i class="fas fa-save"></i> Salvar Configuração
                    </button>
                    <button type="button" class="config-btn config-btn-warning" onclick="resetToDefault()">
                        <i class="fas fa-undo"></i> Resetar para Padrão
                    </button>
                    <button type="button" class="config-btn config-btn-secondary" onclick="closeConfigPopup()">
                        <i class="fas fa-times"></i> Cancelar
                    </button>
                </div>
            </div>
        </div>
    </div>

    <footer class="footer">
        <div class="container">
            <p class="text-center text-muted">© <?php echo date('Y'); ?> Sistema de Comentários</p>
        </div>
    </footer>

    <?php $conn->close(); ?>
    <script>
    // Copiar link ao clicar no campo de link, sem popup
    document.addEventListener('DOMContentLoaded', () => {
        const linkCells = document.querySelectorAll('td.col-link');
        linkCells.forEach(cell => {
            cell.addEventListener('click', () => {
                const link = cell.getAttribute('title') || cell.textContent.trim();
                if (!link) return;
                navigator.clipboard.writeText(link).catch(() => {});
            });
        });
    });
    </script>
    <script>
    // IA assíncrona: envia comentários em lote para backend e atualiza badges
    document.addEventListener('DOMContentLoaded', () => {
        const pendingBadges = Array.from(document.querySelectorAll('.ai-badge.ai-pending'));
        if (pendingBadges.length === 0) return;
        
        // Mostrar que está processando todos
        pendingBadges.forEach(el => {
            el.textContent = 'Analisando...';
            el.classList.add('risk-neutral');
        });
        
        // Coletar todos os comentários
        const comments = pendingBadges.map(el => el.getAttribute('data-comment') || '');
        
        // Enviar TODOS para análise simultaneamente (sem lotes)
        fetch('ia_analyze.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ comments })
        })
        .then(r => r.json())
        .then(data => {
            const results = (data && data.results) ? data.results : [];
            
            // Aplicar todos os resultados de uma vez
            pendingBadges.forEach((el, i) => {
                const res = results[i] || { color: 'risk-neutral', label: 'Sem score', detail: '' };
                el.classList.remove('risk-neutral','risk-green','risk-yellow','risk-orange','risk-red','risk-blue');
                el.classList.add(res.color);
                el.textContent = res.label || 'OK';
                if (res.detail) el.title = res.detail;
                
                const dot = el.parentElement && el.parentElement.querySelector('.ai-dot');
                if (dot) {
                    dot.classList.remove('risk-neutral','risk-green','risk-yellow','risk-orange','risk-red','risk-blue');
                    dot.classList.add(res.color);
                }
            });
        })
        .catch(() => {
            // Em caso de erro, marcar todos como erro
            pendingBadges.forEach(el => {
                el.textContent = 'Erro';
                el.classList.add('risk-neutral');
            });
        });
    });
    
    // Funções para o popup de configuração da IA
    function openConfigPopup() {
        document.getElementById('configPopup').style.display = 'block';
        document.body.style.overflow = 'hidden'; // Previne scroll da página
    }
    
    function closeConfigPopup() {
        document.getElementById('configPopup').style.display = 'none';
        document.body.style.overflow = 'auto'; // Restaura scroll da página
    }
    
    // Fechar popup ao clicar fora dele
    document.getElementById('configPopup').addEventListener('click', function(e) {
        if (e.target === this) {
            closeConfigPopup();
        }
    });
    
    // Fechar popup com ESC
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && document.getElementById('configPopup').style.display === 'block') {
            closeConfigPopup();
        }
    });
    
    function saveConfig() {
        const thresholds = {
            'TOXICITY': parseFloat(document.getElementById('threshold_toxicity').value),
            'SEVERE_TOXICITY': parseFloat(document.getElementById('threshold_severe').value),
            'INSULT': parseFloat(document.getElementById('threshold_insult').value),
            'PROFANITY': parseFloat(document.getElementById('threshold_profanity').value),
            'THREAT': parseFloat(document.getElementById('threshold_threat').value),
            'IDENTITY_ATTACK': parseFloat(document.getElementById('threshold_identity').value)
        };
        
        const negativeWords = document.getElementById('negative_words').value.split(',').map(w => w.trim()).filter(w => w);
        const positiveWords = document.getElementById('positive_words').value.split(',').map(w => w.trim()).filter(w => w);
        
        const timeouts = {
            'main_timeout': parseInt(document.getElementById('timeout_main').value),
            'connect_timeout': parseInt(document.getElementById('timeout_connect').value),
            'fallback_timeout': parseInt(document.getElementById('timeout_fallback').value),
            'fallback_connect_timeout': parseInt(document.getElementById('timeout_fallback_connect').value)
        };
        
        // Enviar para salvar via AJAX
        fetch('save_config_ia.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                thresholds: thresholds,
                negative_words: negativeWords,
                positive_words: positiveWords,
                timeouts: timeouts
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('success', '✅ Configuração salva com sucesso!');
                // Recarregar a página após 1 segundo para aplicar as mudanças
                setTimeout(() => {
                    location.reload();
                }, 1000);
            } else {
                showAlert('danger', '❌ Erro ao salvar configuração: ' + (data.error || 'Erro desconhecido'));
            }
        })
        .catch(error => {
            showAlert('danger', '❌ Erro ao salvar configuração: ' + error.message);
        });
    }
    
    
    function resetToDefault() {
        if (confirm('Deseja resetar todas as configurações para os valores padrão?')) {
            // Resetar thresholds
            document.getElementById('threshold_toxicity').value = '0.05';
            document.getElementById('threshold_severe').value = '0.05';
            document.getElementById('threshold_insult').value = '0.05';
            document.getElementById('threshold_profanity').value = '0.05';
            document.getElementById('threshold_threat').value = '0.05';
            document.getElementById('threshold_identity').value = '0.05';
            
            // Resetar palavras-chave
            document.getElementById('negative_words').value = 'ruim, péssimo, horrível, terrível, merda, porcaria, lixo, fraco, decepcionado, insatisfeito, não recomendo, evitar, problema, defeito, quebrado, danificado, atraso, demora';
            document.getElementById('positive_words').value = 'bom, ótimo, excelente, fantástico, maravilhoso, perfeito, recomendo, satisfeito, qualidade, superou, vale a pena, funciona, rápido, eficiente, prático';
            
            // Resetar timeouts
            document.getElementById('timeout_main').value = '60';
            document.getElementById('timeout_connect').value = '30';
            document.getElementById('timeout_fallback').value = '60';
            document.getElementById('timeout_fallback_connect').value = '30';
        }
    }
    
    function showAlert(type, message) {
        const alertDiv = document.getElementById('configAlert');
        alertDiv.innerHTML = `<div class="config-alert config-alert-${type}">${message}</div>`;
        setTimeout(() => {
            alertDiv.innerHTML = '';
        }, 5000);
    }
    </script>
</body>
</html> 