/**
 * JavaScript Principal - Sistema de Moderação
 */

const API_BASE = 'api';

// Categorias traduzidas
const CATEGORIES = {
    'hate': 'Ódio',
    'hate/threatening': 'Ódio/Ameaçador',
    'harassment': 'Assédio',
    'harassment/threatening': 'Assédio/Ameaçador',
    'self-harm': 'Autolesão',
    'self-harm/intent': 'Autolesão/Intenção',
    'self-harm/instructions': 'Autolesão/Instruções',
    'sexual': 'Sexual',
    'sexual/minors': 'Sexual/Menores',
    'violence': 'Violência',
    'violence/graphic': 'Violência/Gráfico',
    'illicit': 'Ilícito',
    'illicit/violent': 'Ilícito/Violento'
};

/**
 * Formata categoria para exibição
 */
function formatCategory(category) {
    return CATEGORIES[category] || category;
}

/**
 * Formata pontuação para exibição
 */
function formatScore(score) {
    return (score * 100).toFixed(2) + '%';
}

/**
 * Obtém cor baseada na pontuação
 */
function getScoreColor(score) {
    if (score >= 0.7) return '#dc3545'; // Vermelho
    if (score >= 0.4) return '#ffc107'; // Amarelo
    return '#28a745'; // Verde
}

// Variável para controlar se há uma requisição em andamento
let isProcessing = false;
let lastRequestTime = 0;
const MIN_REQUEST_INTERVAL = 2000; // 2 segundos entre requisições

/**
 * Submete formulário de moderação
 */
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('moderationForm');
    if (form) {
        form.addEventListener('submit', handleModeration);
    }
});

/**
 * Processa moderação de comentário
 */
async function handleModeration(e) {
    e.preventDefault();

    // Proteção contra múltiplos cliques
    if (isProcessing) {
        return;
    }

    const form = e.target;
    const submitBtn = document.getElementById('submitBtn');
    const loading = document.getElementById('loading');
    const results = document.getElementById('results');
    const commentInput = document.getElementById('commentInput');
    const multipleInputs = document.getElementById('multipleInputs');

    const comment = commentInput.value.trim();
    if (!comment) {
        alert('Por favor, digite um comentário para moderar.');
        return;
    }

    // Verificar intervalo mínimo entre requisições
    const now = Date.now();
    const timeSinceLastRequest = now - lastRequestTime;
    if (timeSinceLastRequest < MIN_REQUEST_INTERVAL) {
        const waitTime = Math.ceil((MIN_REQUEST_INTERVAL - timeSinceLastRequest) / 1000);
        alert(`Por favor, aguarde ${waitTime} segundo(s) antes de fazer outra requisição.`);
        return;
    }

    // Preparar dados
    let inputData;
    if (multipleInputs.checked) {
        // Múltiplos comentários (um por linha)
        inputData = comment.split('\n').filter(line => line.trim());
        if (inputData.length === 0) {
            alert('Por favor, digite pelo menos um comentário.');
            return;
        }
    } else {
        inputData = comment;
    }

    // Marcar como processando
    isProcessing = true;
    lastRequestTime = Date.now();
    
    // Mostrar loading
    submitBtn.disabled = true;
    loading.style.display = 'block';
    results.style.display = 'none';

    try {
        const response = await fetch(`${API_BASE}/moderate.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ input: inputData })
        });

        const data = await response.json();

        if (!data.success) {
            // Tratamento especial para erro 429
            if (data.error && data.error.includes('429')) {
                throw new Error('Muitas requisições. A API está temporariamente sobrecarregada. Por favor, aguarde alguns segundos e tente novamente.');
            }
            throw new Error(data.error || 'Erro ao processar moderação');
        }

        // Exibir resultados
        displayResults(data.results, Array.isArray(inputData));
        
    } catch (error) {
        console.error('Erro:', error);
        
        // Mostrar erro de forma mais amigável
        const errorMessage = error.message || 'Erro desconhecido ao processar moderação';
        results.style.display = 'block';
        results.innerHTML = `
            <div class="alert alert-danger fade-in">
                <h5><i class="bi bi-exclamation-triangle"></i> Erro na Moderação</h5>
                <p>${escapeHtml(errorMessage)}</p>
                ${errorMessage.includes('429') || errorMessage.includes('Muitas requisições') ? `
                    <hr>
                    <p><strong>Dicas:</strong></p>
                    <ul>
                        <li>Aguarde 30-60 segundos antes de tentar novamente</li>
                        <li>Evite fazer múltiplas requisições rapidamente</li>
                        <li>A API de Moderação tem limites de taxa de requisições</li>
                    </ul>
                ` : ''}
            </div>
        `;
    } finally {
        isProcessing = false;
        submitBtn.disabled = false;
        loading.style.display = 'none';
    }
}

/**
 * Exibe resultados da moderação
 */
function displayResults(results, isMultiple) {
    const resultsDiv = document.getElementById('results');
    resultsDiv.style.display = 'block';
    resultsDiv.innerHTML = '';

    if (isMultiple) {
        // Múltiplos resultados
        results.forEach((result, index) => {
            resultsDiv.innerHTML += createResultHTML(result, index + 1);
        });
    } else {
        // Resultado único
        resultsDiv.innerHTML = createResultHTML(results, null);
    }

    // Scroll para resultados
    resultsDiv.scrollIntoView({ behavior: 'smooth', block: 'start' });
}

/**
 * Cria HTML do resultado
 */
function createResultHTML(result, index) {
    const isFlagged = result.flagged;
    const resultClass = isFlagged ? 'flagged' : 'approved';
    const resultIcon = isFlagged ? 'bi-exclamation-triangle' : 'bi-check-circle';
    const resultText = isFlagged ? 'Comentário Sinalizado' : 'Comentário Aprovado';
    const resultColor = isFlagged ? 'danger' : 'success';

    let html = `
        <div class="moderation-result ${resultClass} fade-in">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <h5>
                    <i class="bi ${resultIcon} text-${resultColor}"></i>
                    ${index ? `Comentário #${index}: ` : ''}${resultText}
                </h5>
                <span class="badge bg-${resultColor}">ID: ${result.comment_id}</span>
            </div>
            
            <div class="comment-content mb-3">
                ${escapeHtml(result.content)}
            </div>

            <div class="mb-3">
                <strong>Categorias Detectadas:</strong>
                <div class="mt-2">
    `;

    // Exibir categorias
    const categories = result.categories || {};
    const scores = result.category_scores || {};
    
    let hasDetected = false;
    for (const [category, detected] of Object.entries(categories)) {
        if (detected) {
            hasDetected = true;
            const score = scores[category] || 0;
            html += `
                <div class="mb-2">
                    <div class="d-flex justify-content-between align-items-center">
                        <span class="category-badge detected">
                            <i class="bi bi-exclamation-circle"></i> ${formatCategory(category)}
                        </span>
                        <strong style="color: ${getScoreColor(score)}">${formatScore(score)}</strong>
                    </div>
                    <div class="score-bar">
                        <div class="score-fill" style="width: ${score * 100}%; background-color: ${getScoreColor(score)}"></div>
                    </div>
                </div>
            `;
        }
    }

    if (!hasDetected) {
        html += '<span class="text-muted">Nenhuma categoria problemática detectada</span>';
    }

    html += `
                </div>
            </div>

            <div class="mt-3">
                <strong>Todas as Categorias:</strong>
                <div class="mt-2">
    `;

    // Exibir todas as categorias com pontuações
    for (const [category, detected] of Object.entries(categories)) {
        const score = scores[category] || 0;
        const badgeClass = detected ? 'detected' : 'not-detected';
        html += `
            <span class="category-badge ${badgeClass}" title="Pontuação: ${formatScore(score)}">
                ${formatCategory(category)}: ${formatScore(score)}
            </span>
        `;
    }

    html += `
                </div>
            </div>
        </div>
    `;

    return html;
}

/**
 * Escapa HTML para prevenir XSS
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Função utilitária para fazer requisições
 */
async function apiRequest(endpoint, method = 'GET', data = null) {
    const options = {
        method: method,
        headers: {
            'Content-Type': 'application/json'
        }
    };

    if (data && method !== 'GET') {
        options.body = JSON.stringify(data);
    }

    const response = await fetch(`${API_BASE}/${endpoint}`, options);
    return await response.json();
}

