<?php
/**
 * API para Listar Comentários
 * Endpoint para obter lista de comentários com resultados de moderação
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../config/database.php';

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        throw new Exception('Método não permitido');
    }

    $db = Database::getInstance();
    $conn = $db->getConnection();

    // Parâmetros opcionais
    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 100;
    $offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;
    $flagged = isset($_GET['flagged']) ? $_GET['flagged'] : null;

    // Construir query
    $where = "WHERE 1=1";
    if ($flagged !== null) {
        $flagged = $flagged === '1' || $flagged === 'true' ? 1 : 0;
        $where .= " AND c.flagged = " . intval($flagged);
    }

    // Buscar comentários
    $sql = "
        SELECT 
            c.id,
            c.content,
            c.flagged,
            c.moderation_id,
            c.created_at
        FROM comments c
        $where
        ORDER BY c.created_at DESC
        LIMIT ? OFFSET ?
    ";

    $stmt = $conn->prepare($sql);
    $stmt->bind_param("ii", $limit, $offset);
    $stmt->execute();
    $result = $stmt->get_result();
    $stmt->close();

    $comments = [];
    while ($row = $result->fetch_assoc()) {
        $commentId = $row['id'];

        // Buscar categorias
        $catStmt = $conn->prepare("
            SELECT category_name, category_detected, category_score
            FROM moderation_categories
            WHERE comment_id = ?
        ");
        $catStmt->bind_param("i", $commentId);
        $catStmt->execute();
        $catResult = $catStmt->get_result();

        $categories = [];
        $categoryScores = [];

        while ($catRow = $catResult->fetch_assoc()) {
            $catName = $catRow['category_name'];
            $categories[$catName] = (bool)$catRow['category_detected'];
            $categoryScores[$catName] = floatval($catRow['category_score']);
        }
        $catStmt->close();

        // Verificar se tem feedback
        $feedbackStmt = $conn->prepare("
            SELECT COUNT(*) as has_feedback
            FROM feedback
            WHERE comment_id = ?
        ");
        $feedbackStmt->bind_param("i", $commentId);
        $feedbackStmt->execute();
        $feedbackResult = $feedbackStmt->get_result();
        $hasFeedback = $feedbackResult->fetch_assoc()['has_feedback'] > 0;
        $feedbackStmt->close();

        $comments[] = [
            'id' => intval($row['id']),
            'content' => $row['content'],
            'flagged' => (bool)$row['flagged'],
            'moderation_id' => $row['moderation_id'],
            'created_at' => $row['created_at'],
            'categories' => $categories,
            'category_scores' => $categoryScores,
            'has_feedback' => $hasFeedback
        ];
    }

    // Contar total
    $countSql = "SELECT COUNT(*) as total FROM comments c $where";
    $countResult = $conn->query($countSql);
    $total = $countResult->fetch_assoc()['total'];

    echo json_encode([
        'success' => true,
        'comments' => $comments,
        'total' => intval($total),
        'limit' => $limit,
        'offset' => $offset
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

