/**
 * JavaScript do Treinamento/Ajustador
 */

let currentCommentId = null;
let allComments = [];

document.addEventListener('DOMContentLoaded', function() {
    loadComments();
    loadAccuracyStats();
});

/**
 * Carrega lista de comentários
 */
async function loadComments() {
    const commentsList = document.getElementById('commentsList');
    if (!commentsList) return;
    
    commentsList.innerHTML = '<div class="text-center py-5"><div class="spinner-border text-primary"></div><p class="mt-2">Carregando comentários...</p></div>';

    try {
        // Aplicar filtros se existirem
        const flagged = document.getElementById('filterFlagged')?.value || '';
        const category = document.getElementById('filterCategory')?.value || '';
        const date = document.getElementById('filterDate')?.value || '';
        
        let url = 'api/comments.php?limit=100';
        if (flagged !== '') url += '&flagged=' + flagged;
        
        const response = await fetch(url);
        const data = await response.json();

        if (data.success) {
            allComments = data.comments;
            
            // Aplicar filtros adicionais no cliente
            let filtered = [...allComments];
            
            if (category !== '') {
                filtered = filtered.filter(c => {
                    return c.categories && c.categories[category] === true;
                });
            }
            
            if (date !== '') {
                filtered = filtered.filter(c => {
                    const commentDate = new Date(c.created_at).toISOString().split('T')[0];
                    return commentDate === date;
                });
            }
            
            displayComments(filtered);
        } else {
            throw new Error(data.error || 'Erro ao carregar comentários');
        }
    } catch (error) {
        console.error('Erro:', error);
        commentsList.innerHTML = `
            <div class="alert alert-danger">
                <i class="bi bi-exclamation-triangle"></i> Erro ao carregar comentários: ${error.message}
            </div>
        `;
    }
}

/**
 * Filtra comentários (recarrega do servidor)
 */
function filterComments() {
    loadComments();
}

/**
 * Exibe comentários na lista
 */
function displayComments(comments) {
    const commentsList = document.getElementById('commentsList');

    if (comments.length === 0) {
        commentsList.innerHTML = '<div class="alert alert-info">Nenhum comentário encontrado.</div>';
        return;
    }

    let html = '';

    comments.forEach(comment => {
        const isFlagged = comment.flagged;
        const flagBadge = isFlagged 
            ? '<span class="badge bg-danger">Sinalizado</span>' 
            : '<span class="badge bg-success">Aprovado</span>';

        const detectedCategories = Object.entries(comment.categories || {})
            .filter(([cat, detected]) => detected)
            .map(([cat]) => formatCategory(cat));

        html += `
            <div class="comment-item fade-in">
                <div class="d-flex justify-content-between align-items-start mb-2">
                    <div>
                        ${flagBadge}
                        <span class="badge bg-secondary ms-2">ID: ${comment.id}</span>
                        <small class="text-muted ms-2">${formatDate(comment.created_at)}</small>
                    </div>
                </div>
                
                <div class="comment-content">
                    ${escapeHtml(comment.content)}
                </div>

                ${detectedCategories.length > 0 ? `
                    <div class="mb-2">
                        <strong>Categorias detectadas:</strong>
                        ${detectedCategories.map(cat => `<span class="badge bg-warning text-dark ms-1">${cat}</span>`).join('')}
                    </div>
                ` : ''}

                <div class="action-buttons mt-3">
                    <button class="btn btn-success btn-action" onclick="openFeedbackModal(${comment.id}, 'good')">
                        <i class="bi bi-check-circle"></i> Marcar como Bom
                    </button>
                    <button class="btn btn-danger btn-action" onclick="openFeedbackModal(${comment.id}, 'bad')">
                        <i class="bi bi-x-circle"></i> Marcar como Ruim
                    </button>
                    <button class="btn btn-info btn-action" onclick="viewDetails(${comment.id})">
                        <i class="bi bi-eye"></i> Ver Detalhes
                    </button>
                </div>
            </div>
        `;
    });

    commentsList.innerHTML = html;
}

/**
 * Abre modal de feedback
 */
function openFeedbackModal(commentId, defaultJudgment) {
    currentCommentId = commentId;
    
    const comment = allComments.find(c => c.id === commentId);
    if (!comment) {
        alert('Comentário não encontrado');
        return;
    }

    // Preencher modal
    document.getElementById('modalCommentText').textContent = comment.content;
    
    const resultHtml = comment.flagged 
        ? '<span class="badge bg-danger">Sinalizado pela moderação</span>'
        : '<span class="badge bg-success">Aprovado pela moderação</span>';
    document.getElementById('modalModerationResult').innerHTML = resultHtml;

    // Definir julgamento padrão
    if (defaultJudgment === 'good') {
        document.getElementById('judgmentGood').checked = true;
    } else {
        document.getElementById('judgmentBad').checked = true;
    }

    // Abrir modal
    const modal = new bootstrap.Modal(document.getElementById('feedbackModal'));
    modal.show();
}

/**
 * Submete feedback
 */
async function submitFeedback() {
    const judgment = document.querySelector('input[name="userJudgment"]:checked').value;
    const notes = document.getElementById('feedbackNotes').value;

    if (!currentCommentId) {
        alert('Erro: ID do comentário não encontrado');
        return;
    }

    try {
        const response = await fetch('api/feedback.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                comment_id: currentCommentId,
                user_judgment: judgment,
                notes: notes
            })
        });

        const data = await response.json();

        if (data.success) {
            alert('Feedback salvo com sucesso!');
            
            // Fechar modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('feedbackModal'));
            modal.hide();

            // Recarregar estatísticas
            loadAccuracyStats();
        } else {
            throw new Error(data.error || 'Erro ao salvar feedback');
        }
    } catch (error) {
        console.error('Erro:', error);
        alert('Erro ao salvar feedback: ' + error.message);
    }
}

/**
 * Carrega estatísticas de precisão
 */
async function loadAccuracyStats() {
    try {
        const response = await apiRequest('stats.php?type=accuracy');
        if (response.success) {
            displayAccuracyStats(response.data);
        }
    } catch (error) {
        console.error('Erro ao carregar estatísticas:', error);
    }
}

/**
 * Exibe estatísticas de precisão
 */
function displayAccuracyStats(data) {
    const statsDiv = document.getElementById('accuracyStats');
    
    statsDiv.innerHTML = `
        <div class="col-md-4">
            <div class="stat-card bg-success text-white">
                <div class="stat-label">Moderações Corretas</div>
                <div class="stat-value">${data.correct}</div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="stat-card bg-danger text-white">
                <div class="stat-label">Moderações Incorretas</div>
                <div class="stat-value">${data.incorrect}</div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="stat-card bg-info text-white">
                <div class="stat-label">Taxa de Precisão</div>
                <div class="stat-value">${data.accuracy_rate}%</div>
                <div class="stat-label">${data.false_positives} FP / ${data.false_negatives} FN</div>
            </div>
        </div>
    `;
}

/**
 * Visualiza detalhes do comentário
 */
function viewDetails(commentId) {
    const comment = allComments.find(c => c.id === commentId);
    if (!comment) {
        alert('Comentário não encontrado');
        return;
    }

    let details = `Comentário ID: ${comment.id}\n\n`;
    details += `Conteúdo:\n${comment.content}\n\n`;
    details += `Status: ${comment.flagged ? 'Sinalizado' : 'Aprovado'}\n\n`;
    details += `Categorias e Pontuações:\n`;

    if (comment.category_scores) {
        for (const [category, score] of Object.entries(comment.category_scores)) {
            details += `- ${formatCategory(category)}: ${formatScore(score)}\n`;
        }
    }

    alert(details);
}

/**
 * Formata data
 */
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleString('pt-BR');
}

// Adicionar event listener para filtros quando o DOM estiver pronto
setTimeout(() => {
    const filterFlagged = document.getElementById('filterFlagged');
    const filterCategory = document.getElementById('filterCategory');
    const filterDate = document.getElementById('filterDate');
    
    if (filterFlagged) filterFlagged.addEventListener('change', filterComments);
    if (filterCategory) filterCategory.addEventListener('change', filterComments);
    if (filterDate) filterDate.addEventListener('change', filterComments);
}, 100);

