<?php
require_once 'config.php';

// Definir variáveis de pesquisa e paginação
$search = isset($_GET['search']) ? $_GET['search'] : '';
$filter_maquina = isset($_GET['maquina']) ? $_GET['maquina'] : '';
$filter_username = isset($_GET['username']) ? $_GET['username'] : '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$records_per_page = 20;
$offset = ($page - 1) * $records_per_page;

// Verificar mensagens de sucesso ou erro
$success_message = isset($_GET['success']) ? true : false;
$error_message = isset($_GET['error']) ? true : false;

// Array para armazenar parâmetros de pesquisa
$where_conditions = [];
$bind_params = [];
$bind_types = "";

// Adicionar condições de pesquisa principal (orderid e link)
if (!empty($search)) {
    $search_term = "%{$search}%";
    $where_conditions[] = "(orderid LIKE ? OR link LIKE ?)";
    $bind_params[] = $search_term;
    $bind_params[] = $search_term;
    $bind_types .= "ss";
}

// Adicionar filtros de máquina e username
if (!empty($filter_maquina)) {
    $maquina_term = "%{$filter_maquina}%";
    $where_conditions[] = "maquina LIKE ?";
    $bind_params[] = $maquina_term;
    $bind_types .= "s";
}

if (!empty($filter_username)) {
    $username_term = "%{$filter_username}%";
    $where_conditions[] = "username LIKE ?";
    $bind_params[] = $username_term;
    $bind_types .= "s";
}

// Construir a consulta SQL base
$sql_count = "SELECT COUNT(*) as total FROM history";
$sql = "SELECT * FROM history";

// Combinar condições WHERE, se existirem
if (!empty($where_conditions)) {
    $sql_count .= " WHERE " . implode(" AND ", $where_conditions);
    $sql .= " WHERE " . implode(" AND ", $where_conditions);
}

// Adicionar ordenação e limite para paginação
$sql .= " ORDER BY data_acao DESC LIMIT ?, ?";

// Preparar e executar consulta de contagem
$stmt_count = $conn->prepare($sql_count);
if (!empty($bind_params)) {
    $stmt_count->bind_param($bind_types, ...$bind_params);
}
$stmt_count->execute();
$result_count = $stmt_count->get_result();
$row_count = $result_count->fetch_assoc();
$total_records = $row_count['total'];
$total_pages = ceil($total_records / $records_per_page);

// Contadores de status
$total_erro = 0;
$total_completo = 0;
$order_id_pesquisado = null;
$link_pesquisado = null;

// Verificar se há uma pesquisa que seja possivelmente um order_id ou link
if (!empty($search)) {
    if (is_numeric(trim($search))) {
        $order_id_pesquisado = trim($search);
        
        // Consultar contagens por status
        $sql_status = "SELECT status, COUNT(*) as count FROM history WHERE orderid = ? GROUP BY status";
        $stmt_status = $conn->prepare($sql_status);
        $stmt_status->bind_param("i", $order_id_pesquisado);
    } else {
        $link_pesquisado = trim($search);
        
        // Consultar contagens por status para link
        $sql_status = "SELECT status, COUNT(*) as count FROM history WHERE link LIKE ? GROUP BY status";
        $stmt_status = $conn->prepare($sql_status);
        $search_term = "%{$link_pesquisado}%";
        $stmt_status->bind_param("s", $search_term);
    }
    
    // Executar a consulta e processar resultados
    if (isset($stmt_status)) {
        $stmt_status->execute();
        $result_status = $stmt_status->get_result();
        
        // Processar resultados de contagem por status
        while ($status_row = $result_status->fetch_assoc()) {
            if (strtolower($status_row['status']) == 'erro') {
                $total_erro = $status_row['count'];
            } elseif (strtolower($status_row['status']) == 'completo') {
                $total_completo = $status_row['count'];
            }
        }
    }
}

// Preparar e executar consulta principal
$stmt = $conn->prepare($sql);
if (!empty($bind_params)) {
    // Adicionar parâmetros de paginação
    $bind_params[] = $offset;
    $bind_params[] = $records_per_page;
    $bind_types .= "ii";
    $stmt->bind_param($bind_types, ...$bind_params);
} else {
    $stmt->bind_param("ii", $offset, $records_per_page);
}
$stmt->execute();
$result = $stmt->get_result();

// Função auxiliar para construir URL de paginação com filtros
function buildPaginationUrl($page_num) {
    $params = [];
    
    if (!empty($_GET['search'])) {
        $params[] = 'search=' . urlencode($_GET['search']);
    }
    
    if (!empty($_GET['maquina'])) {
        $params[] = 'maquina=' . urlencode($_GET['maquina']);
    }
    
    if (!empty($_GET['username'])) {
        $params[] = 'username=' . urlencode($_GET['username']);
    }
    
    $params[] = 'page=' . $page_num;
    
    return '?' . implode('&', $params);
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Histórico de Registros</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <header class="header">
        <div class="container">
            <div class="header-content">
                <h1 class="header-title">
                    <i class="fas fa-history"></i>Histórico Detalhado
                    <?php if ($total_records > 0): ?>
                        <span class="badge"><?php echo $total_records; ?></span>
                    <?php endif; ?>
                </h1>
                <div class="header-actions">
                    <a href="historico_e_pendentes.php" class="btn btn-primary btn-sm me-2">
                        <i class="fas fa-arrow-left"></i>Voltar ao Menu Principal
                    </a>
                    <a href="adicionar.php" class="btn btn-success btn-sm">
                        <i class="fas fa-plus"></i>Novo Registro
                    </a>
                </div>
            </div>
        </div>
    </header>

    <main class="page-container">
        <?php if ($success_message): ?>
            <div class="alert alert-success alert-dismissible">
                <i class="fas fa-check-circle me-2"></i>Todos os registros foram atualizados para o status "Completo" com sucesso!
                <button type="button" class="btn-close" onclick="this.parentElement.style.display='none';" aria-label="Fechar"></button>
            </div>
        <?php endif; ?>
        
        <?php if ($error_message): ?>
            <div class="alert alert-danger alert-dismissible">
                <i class="fas fa-exclamation-circle me-2"></i>Ocorreu um erro ao atualizar o status dos registros.
                <button type="button" class="btn-close" onclick="this.parentElement.style.display='none';" aria-label="Fechar"></button>
            </div>
        <?php endif; ?>
        
        <section class="card">
            <div class="card-header">
                <h2 class="card-title">Filtros</h2>
                <?php if (!empty($search) || !empty($filter_maquina) || !empty($filter_username)): ?>
                    <a href="index.php" class="btn btn-secondary btn-sm">Limpar filtros</a>
                <?php endif; ?>
            </div>
            <div class="card-body">
                <form action="" method="GET" class="filter-form">
                    <div class="filter-group">
                        <label class="form-label" for="search">Order ID / Link</label>
                        <input type="text" name="search" id="search" class="form-control" placeholder="Pesquisar..." value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                    
                    <div class="filter-group">
                        <label class="form-label" for="maquina">Máquina</label>
                        <input type="text" name="maquina" id="maquina" class="form-control" placeholder="Filtrar por máquina" value="<?php echo htmlspecialchars($filter_maquina); ?>">
                    </div>
                    
                    <div class="filter-group">
                        <label class="form-label" for="username">Usuário</label>
                        <input type="text" name="username" id="username" class="form-control" placeholder="Filtrar por usuário" value="<?php echo htmlspecialchars($filter_username); ?>">
                    </div>
                    
                    <div class="filter-group">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-filter"></i>Filtrar
                        </button>
                    </div>
                </form>
                
                <?php if (!empty($search) || !empty($filter_maquina) || !empty($filter_username)): ?>
                    <div class="filter-tags">
                        <?php if (!empty($search)): ?>
                            <div class="filter-tag">
                                <i class="fas fa-search"></i>
                                <?php echo htmlspecialchars($search); ?>
                                <a href="<?php echo '?' . http_build_query(array_merge($_GET, ['search' => null])); ?>" class="remove">
                                    <i class="fas fa-times"></i>
                                </a>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($filter_maquina)): ?>
                            <div class="filter-tag">
                                <i class="fas fa-desktop"></i>
                                <?php echo htmlspecialchars($filter_maquina); ?>
                                <a href="<?php echo '?' . http_build_query(array_merge($_GET, ['maquina' => null])); ?>" class="remove">
                                    <i class="fas fa-times"></i>
                                </a>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($filter_username)): ?>
                            <div class="filter-tag">
                                <i class="fas fa-user"></i>
                                <?php echo htmlspecialchars($filter_username); ?>
                                <a href="<?php echo '?' . http_build_query(array_merge($_GET, ['username' => null])); ?>" class="remove">
                                    <i class="fas fa-times"></i>
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        </section>

        <?php if ($order_id_pesquisado || $link_pesquisado): ?>
            <section class="stats-container">
                <div class="stat-card">
                    <div class="stat-icon primary">
                        <i class="fas fa-list"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $total_records; ?></div>
                        <div class="stat-label">Total de registros</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon danger">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $total_erro; ?></div>
                        <div class="stat-label">Com erro</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon success">
                        <i class="fas fa-check"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-value"><?php echo $total_completo; ?></div>
                        <div class="stat-label">Completos</div>
                    </div>
                </div>
            </section>
            
            <?php if ($order_id_pesquisado): ?>
            <div class="text-center mb-4">
                <form action="completar_todos.php" method="POST" id="completarForm">
                    <input type="hidden" name="order_id" value="<?php echo $order_id_pesquisado; ?>">
                    <button type="button" class="btn btn-success" onclick="confirmarCompletar()">
                        <i class="fas fa-check-circle"></i>Completar Todos
                    </button>
                </form>
            </div>
            <?php endif; ?>
        <?php endif; ?>

        <section class="card">
            <?php if ($result->num_rows > 0): ?>
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th class="col-id">#</th>
                                <th>Order ID</th>
                                <th class="col-link">Link</th>
                                <th class="col-comment">Comentário</th>
                                <th class="col-date">Data</th>
                                <th>Usuário</th>
                                <th>Máquina</th>
                                <th>Perfil</th>
                                <th class="col-status">Status</th>
                                <th class="col-actions">Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($row = $result->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo $row['id']; ?></td>
                                    <td><?php echo $row['orderid'] ? $row['orderid'] : '-'; ?></td>
                                    <td class="col-link">
                                        <a href="<?php echo htmlspecialchars($row['link']); ?>" target="_blank" class="truncate" title="<?php echo htmlspecialchars($row['link']); ?>">
                                            <?php echo htmlspecialchars($row['link']); ?>
                                        </a>
                                    </td>
                                    <td class="col-comment">
                                        <span class="truncate" title="<?php echo htmlspecialchars($row['comment']); ?>">
                                        <?php echo htmlspecialchars($row['comment']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo date('d/m/y H:i', strtotime($row['data_acao'])); ?></td>
                                    <td><?php echo htmlspecialchars($row['username']); ?></td>
                                    <td><?php echo htmlspecialchars($row['maquina']); ?></td>
                                    <td><?php echo htmlspecialchars($row['profile']); ?></td>
                                    <td>
                                        <?php if (!empty($row['status'])): ?>
                                            <?php 
                                                $status_lower = strtolower($row['status']);
                                                $status_class = $status_lower == 'completo' ? 'success' : 
                                                    ($status_lower == 'erro' ? 'danger' : 'neutral'); 
                                            ?>
                                            <span class="status-badge status-badge-<?php echo $status_class; ?>">
                                                <?php echo htmlspecialchars($row['status']); ?>
                                            </span>
                                        <?php else: ?>
                                            -
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="d-flex justify-content-end">
                                            <a href="visualizar.php?id=<?php echo $row['id']; ?>" class="action-icon view" title="Visualizar">
                                            <i class="fas fa-eye"></i>
                                        </a>
                                            <a href="editar.php?id=<?php echo $row['id']; ?>" class="action-icon edit" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        </div>
                                    </td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <i class="fas fa-search"></i>
                    <p>Nenhum registro encontrado</p>
                    <?php if (!empty($search) || !empty($filter_maquina) || !empty($filter_username)): ?>
                        <small class="text-muted">Tente ajustar seus critérios de pesquisa</small>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </section>

        <?php if ($total_pages > 1): ?>
            <nav aria-label="Paginação">
            <ul class="pagination">
                <li class="page-item <?php echo ($page <= 1) ? 'disabled' : ''; ?>">
                        <a class="page-link" href="<?php echo buildPaginationUrl($page - 1); ?>" aria-label="Anterior">
                        <i class="fas fa-chevron-left"></i>
                    </a>
                </li>
                
                <?php
                // Exibir no máximo 5 links de página
                $start_page = max(1, min($page - 2, $total_pages - 4));
                $end_page = min($start_page + 4, $total_pages);
                
                if ($start_page > 1): ?>
                    <li class="page-item">
                        <a class="page-link" href="<?php echo buildPaginationUrl(1); ?>">1</a>
                    </li>
                    <?php if ($start_page > 2): ?>
                        <li class="page-item disabled">
                            <span class="page-link">...</span>
                        </li>
                    <?php endif; ?>
                <?php endif; ?>
                
                <?php for ($i = $start_page; $i <= $end_page; $i++): ?>
                    <li class="page-item <?php echo ($page == $i) ? 'active' : ''; ?>">
                        <a class="page-link" href="<?php echo buildPaginationUrl($i); ?>"><?php echo $i; ?></a>
                    </li>
                <?php endfor; ?>
                
                <?php if ($end_page < $total_pages): ?>
                    <?php if ($end_page < $total_pages - 1): ?>
                        <li class="page-item disabled">
                            <span class="page-link">...</span>
                        </li>
                    <?php endif; ?>
                    <li class="page-item">
                        <a class="page-link" href="<?php echo buildPaginationUrl($total_pages); ?>"><?php echo $total_pages; ?></a>
                    </li>
                <?php endif; ?>
                
                <li class="page-item <?php echo ($page >= $total_pages) ? 'disabled' : ''; ?>">
                        <a class="page-link" href="<?php echo buildPaginationUrl($page + 1); ?>" aria-label="Próximo">
                        <i class="fas fa-chevron-right"></i>
                    </a>
                </li>
            </ul>
            </nav>
        <?php endif; ?>
    </main>

    <footer class="text-center mt-4 mb-4 text-muted">
        <small>&copy; <?php echo date('Y'); ?> - Sistema de Histórico</small>
    </footer>

    <script>
    function confirmarCompletar() {
        if (confirm("Tem certeza que deseja marcar todos os registros desta Order ID como Completo?")) {
            document.getElementById('completarForm').submit();
        }
    }
    </script>
</body>
</html> 