<?php
/**
 * Script de Instalação Completo
 * Cria todas as tabelas necessárias do sistema de moderação
 */

require_once 'config.php';

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Instalação - Sistema de Moderação</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px 0;
        }
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
        .card-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 15px 15px 0 0 !important;
            padding: 20px;
        }
        .step-item {
            padding: 10px;
            margin: 5px 0;
            border-radius: 5px;
            background: #f8f9fa;
        }
        .step-item.success {
            background: #d4edda;
            color: #155724;
        }
        .step-item.error {
            background: #f8d7da;
            color: #721c24;
        }
        .step-item.warning {
            background: #fff3cd;
            color: #856404;
        }
    </style>
</head>
<body>
    <div class="container mt-5">
        <div class="row justify-content-center">
            <div class="col-md-10">
                <div class="card shadow-lg">
                    <div class="card-header text-white">
                        <h4 class="mb-0">
                            <i class="fas fa-database me-2"></i>
                            Instalação do Sistema de Moderação
                        </h4>
                        <small>Banco: <?php echo DB_NAME; ?> | Host: <?php echo DB_HOST; ?></small>
                    </div>
                    <div class="card-body p-4">
                        <?php
                        $allSuccess = true;
                        $steps = [];
                        $errors = [];

                        try {
                            // Verificar conexão
                            if ($conn->connect_error) {
                                throw new Exception("Erro de conexão: " . $conn->connect_error);
                            }
                            $steps[] = ['msg' => 'Conexão com banco de dados estabelecida', 'type' => 'success'];

                            // 1. Verificar se tabela comments existe e sua estrutura
                            $checkComments = $conn->query("SHOW TABLES LIKE 'comments'");
                            $commentsExists = $checkComments && $checkComments->num_rows > 0;
                            
                            if ($commentsExists) {
                                // Verificar se tem a estrutura da IA (campo content)
                                $checkStructure = $conn->query("SHOW COLUMNS FROM comments LIKE 'content'");
                                if ($checkStructure && $checkStructure->num_rows > 0) {
                                    $steps[] = ['msg' => 'Tabela comments já existe com estrutura da IA', 'type' => 'success'];
                                } else {
                                    // Tabela existe mas não tem estrutura da IA - criar tabela separada
                                    $sql1 = "CREATE TABLE IF NOT EXISTS `comments_ia` (
                                        `id` INT AUTO_INCREMENT PRIMARY KEY,
                                        `content` TEXT NOT NULL,
                                        `flagged` BOOLEAN DEFAULT FALSE,
                                        `moderation_id` VARCHAR(255),
                                        `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                                        `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                                        INDEX `idx_flagged` (`flagged`),
                                        INDEX `idx_created_at` (`created_at`)
                                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                                    
                                    if ($conn->query($sql1)) {
                                        $steps[] = ['msg' => 'Tabela comments_ia criada (comments principal já existe)', 'type' => 'success'];
                                    } else {
                                        $steps[] = ['msg' => 'Aviso: Tabela comments existe mas com estrutura diferente. Erro ao criar comments_ia: ' . $conn->error, 'type' => 'warning'];
                                    }
                                }
                            } else {
                                // Criar tabela comments da IA
                                $sql1 = "CREATE TABLE IF NOT EXISTS `comments` (
                                    `id` INT AUTO_INCREMENT PRIMARY KEY,
                                    `content` TEXT NOT NULL,
                                    `flagged` BOOLEAN DEFAULT FALSE,
                                    `moderation_id` VARCHAR(255),
                                    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                                    `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                                    INDEX `idx_flagged` (`flagged`),
                                    INDEX `idx_created_at` (`created_at`)
                                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                                
                                if ($conn->query($sql1)) {
                                    $steps[] = ['msg' => 'Tabela comments criada para IA', 'type' => 'success'];
                                } else {
                                    $steps[] = ['msg' => 'Erro ao criar comments: ' . $conn->error, 'type' => 'error'];
                                    $allSuccess = false;
                                }
                            }

                            // 2. Tabela de categorias de moderação (sem foreign key para evitar conflitos)
                            $sql2 = "CREATE TABLE IF NOT EXISTS `moderation_categories` (
                                `id` INT AUTO_INCREMENT PRIMARY KEY,
                                `comment_id` INT NOT NULL,
                                `category_name` VARCHAR(100) NOT NULL,
                                `category_detected` BOOLEAN DEFAULT FALSE,
                                `category_score` DECIMAL(5,4) DEFAULT 0.0000,
                                INDEX `idx_comment_id` (`comment_id`),
                                INDEX `idx_category_name` (`category_name`),
                                INDEX `idx_detected` (`category_detected`)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                            
                            if ($conn->query($sql2)) {
                                $steps[] = ['msg' => 'Tabela moderation_categories criada/verificada', 'type' => 'success'];
                            } else {
                                $steps[] = ['msg' => 'Erro ao criar moderation_categories: ' . $conn->error, 'type' => 'error'];
                                $allSuccess = false;
                            }

                            // 3. Tabela de feedback (sem foreign key)
                            $sql3 = "CREATE TABLE IF NOT EXISTS `feedback` (
                                `id` INT AUTO_INCREMENT PRIMARY KEY,
                                `comment_id` INT NOT NULL,
                                `is_correct` BOOLEAN NOT NULL COMMENT 'TRUE se a moderação estava correta, FALSE se estava incorreta',
                                `user_judgment` ENUM('good', 'bad') NOT NULL COMMENT 'Julgamento do usuário: bom ou ruim',
                                `notes` TEXT,
                                `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                                INDEX `idx_comment_id` (`comment_id`),
                                INDEX `idx_is_correct` (`is_correct`),
                                INDEX `idx_user_judgment` (`user_judgment`)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                            
                            if ($conn->query($sql3)) {
                                $steps[] = ['msg' => 'Tabela feedback criada/verificada', 'type' => 'success'];
                            } else {
                                $steps[] = ['msg' => 'Erro ao criar feedback: ' . $conn->error, 'type' => 'error'];
                                $allSuccess = false;
                            }

                            // 4. Tabela de histórico de moderações (sem foreign key)
                            $sql4 = "CREATE TABLE IF NOT EXISTS `moderation_history` (
                                `id` INT AUTO_INCREMENT PRIMARY KEY,
                                `comment_id` INT NOT NULL,
                                `moderation_response` TEXT COMMENT 'JSON completo da resposta da API',
                                `api_model` VARCHAR(100),
                                `api_id` VARCHAR(255),
                                `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                                INDEX `idx_comment_id` (`comment_id`),
                                INDEX `idx_created_at` (`created_at`)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                            
                            if ($conn->query($sql4)) {
                                $steps[] = ['msg' => 'Tabela moderation_history criada/verificada', 'type' => 'success'];
                            } else {
                                $steps[] = ['msg' => 'Erro ao criar moderation_history: ' . $conn->error, 'type' => 'error'];
                                $allSuccess = false;
                            }

                            // 5. Tabela de estatísticas
                            $sql5 = "CREATE TABLE IF NOT EXISTS `moderation_stats` (
                                `id` INT AUTO_INCREMENT PRIMARY KEY,
                                `stat_date` DATE NOT NULL,
                                `total_comments` INT DEFAULT 0,
                                `flagged_comments` INT DEFAULT 0,
                                `false_positives` INT DEFAULT 0,
                                `false_negatives` INT DEFAULT 0,
                                `category_counts` TEXT COMMENT 'JSON com contagem por categoria',
                                `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                                UNIQUE KEY `unique_date` (`stat_date`),
                                INDEX `idx_stat_date` (`stat_date`)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                            
                            if ($conn->query($sql5)) {
                                $steps[] = ['msg' => 'Tabela moderation_stats criada/verificada', 'type' => 'success'];
                            } else {
                                $steps[] = ['msg' => 'Erro ao criar moderation_stats: ' . $conn->error, 'type' => 'error'];
                                $allSuccess = false;
                            }

                            // 6. Tabela de thresholds (se não existir)
                            $sql6 = "CREATE TABLE IF NOT EXISTS `moderation_thresholds` (
                                `id` INT AUTO_INCREMENT PRIMARY KEY,
                                `category_name` VARCHAR(100) NOT NULL UNIQUE,
                                `threshold_value` DECIMAL(5,4) DEFAULT 0.5000,
                                `enabled` BOOLEAN DEFAULT TRUE,
                                `description` TEXT,
                                `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                                `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                                INDEX `idx_category_name` (`category_name`),
                                INDEX `idx_enabled` (`enabled`)
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                            
                            if ($conn->query($sql6)) {
                                $steps[] = ['msg' => 'Tabela moderation_thresholds criada/verificada', 'type' => 'success'];
                                
                                // Verificar se a coluna description existe, se não, adicionar
                                $checkDesc = $conn->query("SHOW COLUMNS FROM moderation_thresholds LIKE 'description'");
                                if (!$checkDesc || $checkDesc->num_rows === 0) {
                                    $addDesc = "ALTER TABLE moderation_thresholds ADD COLUMN `description` TEXT AFTER `enabled`";
                                    if ($conn->query($addDesc)) {
                                        $steps[] = ['msg' => 'Coluna description adicionada à moderation_thresholds', 'type' => 'success'];
                                    } else {
                                        $steps[] = ['msg' => 'Aviso: Não foi possível adicionar coluna description: ' . $conn->error, 'type' => 'warning'];
                                    }
                                }
                            } else {
                                $steps[] = ['msg' => 'Erro ao criar moderation_thresholds: ' . $conn->error, 'type' => 'error'];
                                $allSuccess = false;
                            }
                            

                            // 7. Tabela de blacklist (já criada no config.php, mas vamos garantir)
                            $sql7 = "CREATE TABLE IF NOT EXISTS `comment_blacklist` (
                                `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
                                `phrase` TEXT NOT NULL,
                                `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                                PRIMARY KEY (`id`),
                                KEY `idx_phrase` (`phrase`(255))
                            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
                            
                            if ($conn->query($sql7)) {
                                $steps[] = ['msg' => 'Tabela comment_blacklist criada/verificada', 'type' => 'success'];
                            } else {
                                $steps[] = ['msg' => 'Erro ao criar comment_blacklist: ' . $conn->error, 'type' => 'error'];
                                $allSuccess = false;
                            }

                            // Verificar tabelas existentes
                            $tables = ['comments', 'comments_ia', 'moderation_categories', 'feedback', 'moderation_history', 'moderation_stats', 'moderation_thresholds', 'comment_blacklist'];
                            $existingTables = [];
                            
                            foreach ($tables as $table) {
                                $result = $conn->query("SHOW TABLES LIKE '$table'");
                                if ($result && $result->num_rows > 0) {
                                    $existingTables[] = $table;
                                }
                            }

                            // Exibir resultados
                            if ($allSuccess) {
                                echo '<div class="alert alert-success">';
                                echo '<h5><i class="fas fa-check-circle me-2"></i> Instalação Concluída com Sucesso!</h5>';
                                echo '<p>Todas as tabelas foram criadas/verificadas no banco de dados.</p>';
                                echo '</div>';

                                echo '<div class="alert alert-info">';
                                echo '<h6><i class="fas fa-list me-2"></i> Tabelas no banco:</h6>';
                                echo '<ul class="mb-0">';
                                foreach ($existingTables as $table) {
                                    echo '<li><code>' . $table . '</code></li>';
                                }
                                echo '</ul>';
                                echo '</div>';

                                echo '<div class="alert alert-warning">';
                                echo '<h6><i class="fas fa-info-circle me-2"></i> Próximos passos:</h6>';
                                echo '<ol class="mb-0">';
                                echo '<li>Configure sua chave da API OpenAI em <code>IA/config/openai.php</code></li>';
                                echo '<li>Acesse <a href="index.php">index.php</a> para começar a usar o sistema</li>';
                                echo '<li>Acesse <a href="pendentes.php">pendentes.php</a> para moderar comentários</li>';
                                echo '</ol>';
                                echo '</div>';

                                echo '<div class="d-grid gap-2 d-md-flex justify-content-md-center">';
                                echo '<a href="index.php" class="btn btn-primary btn-lg"><i class="fas fa-home me-2"></i>Ir para o Sistema</a>';
                                echo '<a href="pendentes.php" class="btn btn-success btn-lg"><i class="fas fa-comments me-2"></i>Ver Comentários Pendentes</a>';
                                echo '</div>';
                            } else {
                                throw new Exception('Alguns erros ocorreram durante a instalação');
                            }

                        } catch (Exception $e) {
                            echo '<div class="alert alert-danger">';
                            echo '<h5><i class="fas fa-exclamation-triangle me-2"></i> Erro na Instalação</h5>';
                            echo '<p>' . htmlspecialchars($e->getMessage()) . '</p>';
                            echo '</div>';
                        }

                        // Exibir log de passos
                        if (!empty($steps)) {
                            echo '<hr>';
                            echo '<h6><i class="fas fa-tasks me-2"></i> Log de Instalação:</h6>';
                            echo '<div class="mt-3">';
                            foreach ($steps as $step) {
                                $icon = $step['type'] === 'success' ? 'check-circle' : ($step['type'] === 'error' ? 'times-circle' : 'exclamation-triangle');
                                $iconClass = $step['type'] === 'success' ? 'text-success' : ($step['type'] === 'error' ? 'text-danger' : 'text-warning');
                                echo '<div class="step-item ' . $step['type'] . '">';
                                echo '<i class="fas fa-' . $icon . ' me-2 ' . $iconClass . '"></i>';
                                echo htmlspecialchars($step['msg']);
                                echo '</div>';
                            }
                            echo '</div>';
                        }
                        ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>

