<?php
require_once 'config.php';

// Função auxiliar para exibir texto com emojis corretamente
function exibir_texto($texto) {
    return htmlspecialchars($texto, ENT_QUOTES, 'UTF-8');
}

// Função para adicionar comentário à blacklist
function addToBlacklist($conn, $comment) {
    $commentTrimmed = trim($comment);
    if (empty($commentTrimmed)) {
        return false;
    }
    
    // Verificar se já existe na blacklist
    $checkStmt = $conn->prepare("SELECT id FROM comment_blacklist WHERE LOWER(TRIM(phrase)) = LOWER(?) LIMIT 1");
    if (!$checkStmt) {
        return false;
    }
    
    $checkStmt->bind_param("s", $commentTrimmed);
    $checkStmt->execute();
    $result = $checkStmt->get_result();
    
    // Se já existe, não adicionar novamente
    if ($result->num_rows > 0) {
        $checkStmt->close();
        return true;
    }
    $checkStmt->close();
    
    // Adicionar à blacklist
    $insertStmt = $conn->prepare("INSERT INTO comment_blacklist (phrase) VALUES (?)");
    if (!$insertStmt) {
        return false;
    }
    
    $insertStmt->bind_param("s", $commentTrimmed);
    $success = $insertStmt->execute();
    $insertStmt->close();
    
    return $success;
}

// Lidar com exclusões
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id'], $_POST['source'])) {
    $delete_id = intval($_POST['delete_id']);
    $source = $_POST['source'];

    if (in_array($source, ['comments', 'commentsaleatorio'])) {
        // Buscar o comentário antes de deletar para adicionar à blacklist
        $selectStmt = $conn->prepare("SELECT comment FROM $source WHERE id = ?");
        if ($selectStmt) {
            $selectStmt->bind_param('i', $delete_id);
            $selectStmt->execute();
            $result = $selectStmt->get_result();
            if ($row = $result->fetch_assoc()) {
                addToBlacklist($conn, $row['comment']);
            }
            $selectStmt->close();
        }
        // Deletar o comentário
        $delete_sql = "DELETE FROM $source WHERE id = ?";
        $stmt = $conn->prepare($delete_sql);
        if ($stmt) {
            $stmt->bind_param('i', $delete_id);
            $stmt->execute();
            $stmt->close();
        }
    }
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Lidar com exclusões de todos com mesmo link
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_link'], $_POST['source'])) {
    $delete_link = $_POST['delete_link'];
    $source = $_POST['source'];

    if (in_array($source, ['comments', 'commentsaleatorio'])) {
        // Buscar todos os comentários com esse link para adicionar à blacklist
        $selectStmt = $conn->prepare("SELECT comment FROM $source WHERE link = ?");
        if ($selectStmt) {
            $selectStmt->bind_param('s', $delete_link);
            $selectStmt->execute();
            $result = $selectStmt->get_result();
            while ($row = $result->fetch_assoc()) {
                addToBlacklist($conn, $row['comment']);
            }
            $selectStmt->close();
        }
        // Deletar os comentários
        $delete_sql = "DELETE FROM $source WHERE link = ?";
        $stmt = $conn->prepare($delete_sql);
        if ($stmt) {
            $stmt->bind_param('s', $delete_link);
            $stmt->execute();
            $stmt->close();
        }
    }
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Lidar com exclusão em massa (selecionados)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'bulk_delete') {
    $ids = isset($_POST['ids']) && is_array($_POST['ids']) ? array_map('intval', $_POST['ids']) : [];
    $sources = isset($_POST['sources']) && is_array($_POST['sources']) ? $_POST['sources'] : [];
    foreach ($ids as $oneId) {
        $tbl = isset($sources[$oneId]) ? preg_replace('/[^a-zA-Z0-9_]/', '', $sources[$oneId]) : '';
        if (!in_array($tbl, ['comments', 'commentsaleatorio'])) { continue; }
        // Buscar o comentário antes de deletar para adicionar à blacklist
        $selectStmt = $conn->prepare("SELECT comment FROM `$tbl` WHERE id = ?");
        if ($selectStmt) {
            $selectStmt->bind_param('i', $oneId);
            $selectStmt->execute();
            $result = $selectStmt->get_result();
            if ($row = $result->fetch_assoc()) {
                addToBlacklist($conn, $row['comment']);
            }
            $selectStmt->close();
        }
        // Deletar o comentário
        $stmt = $conn->prepare("DELETE FROM `$tbl` WHERE id = ?");
        if ($stmt) { $stmt->bind_param('i', $oneId); $stmt->execute(); $stmt->close(); }
    }
    header("Location: " . $_SERVER['PHP_SELF']);
    exit();
}

// Filtro de aprovação: mostrar apenas não aprovados
$approved_condition = "approved = 0";

// Configurar filtro de gênero
$filter_gender = isset($_GET['gender']) ? intval($_GET['gender']) : 0; // 0 = todos, 1 = masculino, 2 = feminino, 3 = unissex

// Configurar paginação
$limit = 50;
$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
$offset = ($page - 1) * $limit;

// Construir condições por tabela (com approved = 0)
$where_comments = "WHERE $approved_condition";
if ($filter_gender > 0) {
    if ($filter_gender == 4) {
        $where_comments = "WHERE gender IS NULL AND $approved_condition";
    } else {
        $where_comments = "WHERE gender = $filter_gender AND $approved_condition";
    }
}

$where_random = "WHERE $approved_condition";
if ($filter_gender > 0) {
    if ($filter_gender == 4) {
        // A tabela commentsaleatorio não possui gender NULL nessa visão de pendentes
        $where_random = "WHERE 1=0";
    } else {
        $where_random = "WHERE gender = $filter_gender AND $approved_condition";
    }
}

// Consultar dados combinados para pendentes
$sql_pendente = "
    SELECT id, order_id, link, comment, gender, 'comments' AS source, 1 AS priority
    FROM comments
    $where_comments
    UNION ALL
    SELECT id, order_id, link, comment, gender, 'commentsaleatorio' AS source, 2 AS priority
    FROM commentsaleatorio
    $where_random
    ORDER BY priority ASC, id ASC
    LIMIT $limit OFFSET $offset
";
$result_pendente = $conn->query($sql_pendente);

// Contar total de registros pendentes por tipo de fonte (apenas não aprovados)
if ($filter_gender == 4) {
    $total_personalizado = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender IS NULL AND $approved_condition")->fetch_assoc()['total'];
    $total_aleatorio = 0; // A tabela commentsaleatorio não tem registros sem gênero (gender é NULL)
} else {
    $total_personalizado = $conn->query("SELECT COUNT(*) AS total FROM comments " . ($where_comments ? $where_comments : ""))->fetch_assoc()['total'];
    $total_aleatorio = $conn->query("SELECT COUNT(*) AS total FROM commentsaleatorio " . ($where_random ? $where_random : ""))->fetch_assoc()['total'];
}
$total_pendentes = $total_personalizado + $total_aleatorio;

// Contar totais por gênero (apenas não aprovados)
$total_masculino = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender = 1 AND $approved_condition")->fetch_assoc()['total'] +
                  $conn->query("SELECT COUNT(*) AS total FROM commentsaleatorio WHERE gender = 1 AND $approved_condition")->fetch_assoc()['total'];
$total_feminino = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender = 2 AND $approved_condition")->fetch_assoc()['total'] +
                 $conn->query("SELECT COUNT(*) AS total FROM commentsaleatorio WHERE gender = 2 AND $approved_condition")->fetch_assoc()['total'];
$total_unissex = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender = 3 AND $approved_condition")->fetch_assoc()['total'] +
                $conn->query("SELECT COUNT(*) AS total FROM commentsaleatorio WHERE gender = 3 AND $approved_condition")->fetch_assoc()['total'];

// Calcular total de comentários sem gênero definido (apenas não aprovados)
$total_indefinido = $conn->query("SELECT COUNT(*) AS total FROM comments WHERE gender IS NULL AND $approved_condition")->fetch_assoc()['total'];

// Função auxiliar para construir URL de paginação
function buildPaginationUrl($page_num) {
    $params = [];
    
    foreach ($_GET as $key => $value) {
        if ($key !== 'page') {
            $params[$key] = $value;
        }
    }
    
    $params['page'] = $page_num;
    
    return '?' . http_build_query($params);
}

// Função para criar links de filtro por gênero
function buildGenderFilterUrl($gender_id) {
    $params = [];
    
    foreach ($_GET as $key => $value) {
        if ($key !== 'gender' && $key !== 'page') {
            $params[$key] = $value;
        }
    }
    
    if ($gender_id > 0) {
        $params['gender'] = $gender_id;
    }
    
    $query_string = http_build_query($params);
    return empty($query_string) ? '?' : '?' . $query_string;
}

// Função para exibir nome do gênero
function getNomeGenero($gender_id) {
    if ($gender_id === null) {
        return "Não definido";
    }
    
    switch ($gender_id) {
        case 1:
            return "Masculino";
        case 2:
            return "Feminino";
        case 3:
            return "Unissex";
        default:
            return "Não definido";
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
    <title>Comentários</title>
    <link rel="icon" href="icon.png" type="image/png">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="popup_config.css">
    <style>
        
        .totals {
            display: flex;
            gap: 1rem;
            margin-bottom: 1rem;
        }
        
        .totals span {
            padding: 0.5rem 1rem;
            background-color: var(--color-surface-variant);
            border-radius: var(--border-radius-md);
            font-size: var(--font-size-sm);
            color: var(--color-text-secondary);
        }
        
        .total-link {
            text-decoration: none;
            transition: all 0.2s ease;
        }
        
        .total-link span {
            padding: 0.5rem 1rem;
            background-color: var(--color-surface-variant);
            border-radius: var(--border-radius-md);
            font-size: var(--font-size-sm);
            color: var(--color-text-secondary);
            transition: all 0.2s ease;
        }
        
        .total-link.active span {
            background-color: var(--color-primary-light);
            color: var(--color-primary);
            font-weight: 600;
        }
        
        .total-link:hover span {
            background-color: var(--color-primary-light);
            color: var(--color-primary);
        }
        
        .main-menu {
            display: flex;
            background-color: var(--color-surface);
            border-radius: var(--border-radius-lg);
            box-shadow: var(--shadow-md);
            overflow: hidden;
            margin-bottom: 2rem;
        }
        
        .main-menu a {
            padding: 1rem 1.5rem;
            color: var(--color-text-secondary);
            text-decoration: none;
            font-weight: 600;
            display: flex;
            align-items: center;
            transition: all var(--transition-base);
            border-bottom: 3px solid transparent;
        }
        
        .main-menu a.active {
            color: var(--color-primary);
            background-color: var(--color-primary-light);
            border-bottom-color: var(--color-primary);
        }
        
        .main-menu a:hover:not(.active) {
            background-color: var(--color-surface-variant);
            color: var(--color-text-primary);
        }
        
        .main-menu i {
            margin-right: 0.5rem;
        }

        @media (min-width: 1200px) {
            .table td.col-link, 
            .table td.col-comment {
                padding-right: 10px;
            }
            
            .table .truncate {
                width: 100%;
                display: block;
            }
        }
        
        .col-gender {
            width: 100px;
            text-align: center;
        }
        
        td.col-gender {
            font-size: 0.9em;
        }
        
        .action-icon.delete-all {
            background-color: #dc3545;
            margin-left: 5px;
        }
        
        .action-icon.delete-all:hover {
            background-color: #bd2130;
        }
        
        .ai-badge.clickable {
            cursor: pointer;
        }

        .ai-modal {
            position: fixed;
            inset: 0;
            background: rgba(0, 0, 0, 0.65);
            display: none;
            align-items: center;
            justify-content: center;
            padding: 1rem;
            z-index: 2000;
        }

        .ai-modal.open {
            display: flex;
        }

        .ai-modal-content {
            background: #fff;
            border-radius: 14px;
            max-width: 820px;
            width: 100%;
            max-height: 90%;
            overflow-y: auto;
            padding: 24px;
            position: relative;
            box-shadow: 0 20px 45px rgba(0, 0, 0, 0.25);
        }

        .ai-modal-close {
            position: absolute;
            top: 12px;
            right: 12px;
            border: none;
            background: transparent;
            font-size: 1.6rem;
            cursor: pointer;
            color: #343a40;
        }

        .ai-detail-comment {
            background: #f8f9fa;
            padding: 12px;
            border-radius: 8px;
            font-family: 'Segoe UI', system-ui, sans-serif;
            white-space: pre-wrap;
            margin-bottom: 16px;
        }

        .ai-detail-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 12px;
        }

        .ai-detail-header h3 {
            margin: 0;
            font-size: 1.2rem;
        }

        .ai-detail-badges {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
        }

        .ai-category-badge {
            padding: 6px 10px;
            border-radius: 999px;
            font-size: 0.85rem;
            background: #e9ecef;
            color: #212529;
            border: 1px solid transparent;
        }

        .ai-category-badge.detected {
            background: #ffc107;
            color: #14213d;
            font-weight: 600;
            border-color: #f0ad4e;
        }

        .ai-category-badge.not-detected {
            opacity: 0.65;
        }

        .ai-score-bar {
            height: 6px;
            background: #e9ecef;
            border-radius: 3px;
            overflow: hidden;
            margin-top: 4px;
        }

        .ai-score-fill {
            height: 100%;
            border-radius: 3px;
        }
    </style>
    <script>
        document.addEventListener('DOMContentLoaded', () => {
            const copyLinkCells = document.querySelectorAll('td.col-link');
            const copyCommentCells = document.querySelectorAll('td.col-comment');

            copyLinkCells.forEach(cell => {
                cell.addEventListener('click', () => {
                    const originalLink = cell.getAttribute('title') || cell.textContent.trim();
                    navigator.clipboard.writeText(originalLink).then(() => {
                        cell.classList.add('copied');
                        setTimeout(() => {
                            cell.classList.remove('copied');
                        }, 1000);
                    }).catch(err => {
                        console.error('Erro ao copiar o link: ', err);
                    });
                });
            });

            copyCommentCells.forEach(cell => {
                cell.addEventListener('click', () => {
                    const commentText = cell.getAttribute('title') || cell.textContent.trim();
                    navigator.clipboard.writeText(commentText).then(() => {
                        cell.classList.add('copied');
                        setTimeout(() => {
                            cell.classList.remove('copied');
                        }, 1000);
                    }).catch(err => {
                        console.error('Erro ao copiar o comentário: ', err);
                    });
                });
            });
        });
    </script>
</head>
<body>
    <main class="page-container">
        <?php
        // Consultar métricas gerais
        // Total de ações hoje
        $current_date = date('Y-m-d');
        $total_acoes_hoje = $conn->query("
            SELECT COALESCE(SUM(actions_today), 0) AS total 
            FROM (
                SELECT SUM(actions_today) AS actions_today FROM comentario01 WHERE DATE(last_action_date) = '$current_date'
                UNION ALL
                SELECT SUM(actions_today) AS actions_today FROM comentario02 WHERE DATE(last_action_date) = '$current_date'
                UNION ALL
                SELECT SUM(actions_today) AS actions_today FROM comentario03 WHERE DATE(last_action_date) = '$current_date'
            ) AS totals
        ")->fetch_assoc()['total'] ?? 0;
        
        // Total de ações ontem
        $yesterday_date = date('Y-m-d', strtotime('-1 day'));
        $acoes_ontem = $conn->query("
            SELECT COUNT(*) AS total 
            FROM history 
            WHERE DATE(data_acao) = '$yesterday_date'
        ")->fetch_assoc()['total'] ?? 0;
        
        // Ações na última hora (60 minutos exatos)
        $acoes_ultima_hora = $conn->query("
            SELECT COUNT(*) AS total 
            FROM history 
            WHERE data_acao > DATE_SUB(NOW(), INTERVAL 60 MINUTE)
        ")->fetch_assoc()['total'] ?? 0;
        
        // Data da última ação
        $ultima_acao = $conn->query("
            SELECT MAX(last_action_date) AS last_date FROM (
                SELECT MAX(last_action_date) AS last_action_date FROM comentario01
                UNION ALL
                SELECT MAX(last_action_date) AS last_action_date FROM comentario02
                UNION ALL
                SELECT MAX(last_action_date) AS last_action_date FROM comentario03
            ) AS dates
        ")->fetch_assoc()['last_date'] ?? 'N/A';
        
        // Formatar data da última ação
        $ultima_acao_formatada = $ultima_acao != 'N/A' ? date('d/m/Y H:i:s', strtotime($ultima_acao)) : 'N/A';
        
        // Totais globais (pendentes para aprovar e aprovados) para o cabeçalho
        $total_pendentes_aprovar_header = $conn->query("
            SELECT 
                (SELECT COUNT(*) FROM comments WHERE approved = 0) + 
                (SELECT COUNT(*) FROM commentsaleatorio WHERE approved = 0) AS total
        ")->fetch_assoc()['total'] ?? 0;
        $total_aprovados_header = $conn->query("
            SELECT 
                (SELECT COUNT(*) FROM comments WHERE approved = 1) + 
                (SELECT COUNT(*) FROM commentsaleatorio WHERE approved = 1) AS total
        ")->fetch_assoc()['total'] ?? 0;

        // Calcular previsão para zerar (baseada nos aprovados prontos)
        $previsao_zerar = 'N/A';
        if ($acoes_ultima_hora > 0 && $total_aprovados_header > 0) {
            $minutos_totais = ceil(($total_aprovados_header / $acoes_ultima_hora) * 60);
            if ($minutos_totais < 60) {
                $previsao_zerar = $minutos_totais . ' min';
            } else {
                $horas = floor($minutos_totais / 60);
                $minutos = $minutos_totais % 60;
                if ($horas < 24) {
                    $previsao_zerar = $horas . 'h ' . $minutos . 'min';
                } else {
                    $dias = floor($horas / 24);
                    $horas_restantes = $horas % 24;
                    $previsao_zerar = $dias . ' dias, ' . $horas_restantes . 'h ' . $minutos . 'min';
                }
            }
        }
        ?>
        
        <div class="metrics-summary">
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo number_format($total_acoes_hoje, 0, ',', '.'); ?></div>
                    <div class="metric-label">Ações Hoje <span style="font-size: 0.8em; opacity: 0.7;">(Ontem: <?php echo number_format($acoes_ontem, 0, ',', '.'); ?>)</span></div>
                </div>
            </div>
            
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-hourglass-half"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo number_format($total_pendentes_aprovar_header, 0, ',', '.'); ?></div>
                    <div class="metric-label">Pendentes p/ aprovar</div>
                </div>
            </div>
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-check-circle"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo number_format($total_aprovados_header, 0, ',', '.'); ?></div>
                    <div class="metric-label">Aprovados (prontos)</div>
                </div>
            </div>
            
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-tachometer-alt"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo number_format($acoes_ultima_hora, 0, ',', '.'); ?></div>
                    <div class="metric-label">Ações na última hora</div>
                </div>
            </div>
            
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-hourglass-end"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo $previsao_zerar; ?></div>
                    <div class="metric-label">Previsão para zerar</div>
                </div>
            </div>
            
            <div class="metric-item">
                <div class="metric-icon">
                    <i class="fas fa-calendar-check"></i>
                </div>
                <div class="metric-info">
                    <div class="metric-value"><?php echo $ultima_acao_formatada; ?></div>
                    <div class="metric-label">Última Ação</div>
                </div>
            </div>
        </div>
        
        <div class="main-menu">
            <a href="index.php" class="menu-item">
                <i class="fas fa-history"></i>Histórico
            </a>
            <a href="pendentes.php" class="menu-item active">
                <i class="fas fa-clock"></i>Pendentes
            </a>
            <a href="aprovados.php" class="menu-item">
                <i class="fas fa-check-circle"></i>Aprovados
            </a>
            <a href="maquinas.php" class="menu-item">
                <i class="fas fa-desktop"></i>Máquinas
            </a>
            <a href="controlador.php" class="menu-item">
                <i class="fas fa-cogs"></i>Controlador
            </a>
        </div>

        <div class="totals">
            <span>Total Personalizado: <?php echo $total_personalizado; ?></span>
            <span>Total Aleatório: <?php echo $total_aleatorio; ?></span>
            <a href="<?php echo buildGenderFilterUrl(0); ?>" class="total-link <?php echo $filter_gender == 0 ? 'active' : ''; ?>">
                <span>Total Geral: <?php echo $total_pendentes; ?></span>
            </a>
            <a href="<?php echo buildGenderFilterUrl(1); ?>" class="total-link <?php echo $filter_gender == 1 ? 'active' : ''; ?>">
                <span>Total Masculino: <?php echo $total_masculino; ?></span>
            </a>
            <a href="<?php echo buildGenderFilterUrl(2); ?>" class="total-link <?php echo $filter_gender == 2 ? 'active' : ''; ?>">
                <span>Total Feminino: <?php echo $total_feminino; ?></span>
            </a>
            <a href="<?php echo buildGenderFilterUrl(3); ?>" class="total-link <?php echo $filter_gender == 3 ? 'active' : ''; ?>">
                <span>Total Unissex: <?php echo $total_unissex; ?></span>
            </a>
            <?php if ($total_indefinido > 0): ?>
            <a href="<?php echo buildGenderFilterUrl(4); ?>" class="total-link <?php echo $filter_gender == 4 ? 'active' : ''; ?>">
                <span>Não definido: <?php echo $total_indefinido; ?></span>
            </a>
            <?php endif; ?>
        </div>

        <div class="card">
          
            <div class="card-body">
                <?php if (isset($result_pendente) && $result_pendente->num_rows > 0): ?>
                    <div class="table-responsive">
                        <div class="bulk-actions" style="margin-bottom:10px; display:flex; gap:8px; align-items:center;">
                            <label style="display:flex; align-items:center; gap:6px; cursor:pointer;">
                                <input type="checkbox" id="selectAll">
                                <span>Selecionar todos</span>
                            </label>
                            <button type="button" id="bulkApprove" class="btn btn-primary btn-sm">
                                <i class="fas fa-check"></i>Aprovar Selecionados
                            </button>
                            <button type="button" id="bulkDelete" class="btn btn-secondary btn-sm">
                                <i class="fas fa-trash-alt"></i>Excluir Selecionados
                            </button>
                        </div>
                        <table class="table">
                            <thead>
                                <tr>
                                    <th style="width:36px;"><input type="checkbox" id="selectAllHeader"></th>
                                    <th class="col-order-id">Order ID</th>
                                    <th class="col-link">Link</th>
                                    <th class="col-comment">Comentário</th>
                                    <th style="width:140px;">IA</th>
                                    <th class="col-gender">Gênero</th>
                                    <th class="col-actions">Ações</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($row = $result_pendente->fetch_assoc()): ?>
                                    <tr>
                                        <td>
                                            <input type="checkbox" class="row-check" value="<?php echo $row['id']; ?>" data-source="<?php echo exibir_texto($row['source']); ?>">
                                        </td>
                                        <td><?php echo exibir_texto($row['order_id']); ?></td>
                                        <td class="col-link" title="<?php echo exibir_texto($row['link']); ?>">
                                            <span class="truncate"><?php echo exibir_texto($row['link']); ?></span>
                                        </td>
                                        <td class="col-comment" title="<?php echo exibir_texto($row['comment']); ?>">
                                            <span class="truncate"><?php echo exibir_texto($row['comment']); ?></span>
                                        </td>
                                        <td>
                                            <div class="ai-cell">
                                                <span class="ai-dot risk-neutral"></span>
                                                <span class="ai-badge risk-neutral ai-pending" data-comment="<?php echo htmlspecialchars($row['comment']); ?>">Analisando...</span>
                                            </div>
                                        </td>
                                        <td class="col-gender"><?php echo getNomeGenero($row['gender']); ?></td>
                                        <td class="col-actions">
                                            <form method="POST" action="approve.php" style="margin: 0; display: inline-block;">
                                                <input type="hidden" name="id" value="<?php echo $row['id']; ?>">
                                                <input type="hidden" name="source" value="<?php echo exibir_texto($row['source']); ?>">
                                                <input type="hidden" name="action" value="approve">
                                                <button type="submit" class="action-icon view" title="Aprovar">
                                                    <i class="fas fa-check"></i>
                                                </button>
                                            </form>
                                            <form method="POST" action="approve.php" style="margin: 0; display: inline-block;">
                                                <input type="hidden" name="link" value="<?php echo exibir_texto($row['link']); ?>">
                                                <input type="hidden" name="source" value="<?php echo exibir_texto($row['source']); ?>">
                                                <input type="hidden" name="action" value="approve_link">
                                                <button type="submit" class="action-icon view" title="Aprovar todos com mesmo link">
                                                    <i class="fas fa-check-double"></i>
                                                </button>
                                            </form>
                                            
                                            <form method="POST" style="margin: 0; display: inline-block;">
                                                <input type="hidden" name="delete_id" value="<?php echo $row['id']; ?>">
                                                <input type="hidden" name="source" value="<?php echo exibir_texto($row['source']); ?>">
                                                <button type="submit" class="action-icon delete" title="Excluir">
                                                    <i class="fas fa-trash-alt"></i>
                                                </button>
                                            </form>
                                            <form method="POST" style="margin: 0; display: inline-block;">
                                                <input type="hidden" name="delete_link" value="<?php echo exibir_texto($row['link']); ?>">
                                                <input type="hidden" name="source" value="<?php echo exibir_texto($row['source']); ?>">
                                                <button type="submit" class="action-icon delete-all" title="Excluir todos com mesmo link" onclick="return confirm('Tem certeza que deseja excluir todos os comentários com este mesmo link?');">
                                                    <i class="fas fa-dumpster"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                    <script>
                        (function(){
                            const selectAll = document.getElementById('selectAll');
                            const selectAllHeader = document.getElementById('selectAllHeader');
                            const rowChecks = () => Array.from(document.querySelectorAll('.row-check'));
                            const setAll = (checked) => rowChecks().forEach(cb => { cb.checked = checked; });
                            const syncHeader = () => {
                                const cbs = rowChecks();
                                const all = cbs.length > 0 && cbs.every(cb => cb.checked);
                                const any = cbs.some(cb => cb.checked);
                                if (selectAllHeader) {
                                    selectAllHeader.checked = all;
                                    selectAllHeader.indeterminate = !all && any;
                                }
                                if (selectAll) {
                                    selectAll.checked = all;
                                    selectAll.indeterminate = !all && any;
                                }
                            };
                            if (selectAll) selectAll.addEventListener('change', e => { setAll(e.target.checked); syncHeader(); });
                            if (selectAllHeader) selectAllHeader.addEventListener('change', e => { setAll(e.target.checked); syncHeader(); });
                            document.addEventListener('change', e => { if (e.target.classList && e.target.classList.contains('row-check')) syncHeader(); });

                            function submitBulk(action){
                                const selected = rowChecks().filter(cb => cb.checked);
                                if (selected.length === 0) { alert('Selecione pelo menos um item.'); return; }
                                const form = document.createElement('form');
                                form.method = 'POST';
                                form.action = 'approve.php';
                                form.style.display = 'none';
                                const inpAction = document.createElement('input');
                                inpAction.type = 'hidden'; inpAction.name = 'action'; inpAction.value = 'bulk';
                                form.appendChild(inpAction);
                                const inpBulk = document.createElement('input');
                                inpBulk.type = 'hidden'; inpBulk.name = 'bulk_action'; inpBulk.value = action;
                                form.appendChild(inpBulk);
                                selected.forEach(cb => {
                                    const idInput = document.createElement('input');
                                    idInput.type = 'hidden'; idInput.name = 'ids[]'; idInput.value = cb.value;
                                    form.appendChild(idInput);
                                    const srcInput = document.createElement('input');
                                    srcInput.type = 'hidden'; srcInput.name = `sources[${cb.value}]`; srcInput.value = cb.getAttribute('data-source');
                                    form.appendChild(srcInput);
                                });
                                document.body.appendChild(form);
                                form.submit();
                            }
                            function submitBulkDelete(){
                                const selected = rowChecks().filter(cb => cb.checked);
                                if (selected.length === 0) { alert('Selecione pelo menos um item.'); return; }
                                const form = document.createElement('form');
                                form.method = 'POST';
                                form.action = '';
                                form.style.display = 'none';
                                const inpAction = document.createElement('input');
                                inpAction.type = 'hidden'; inpAction.name = 'action'; inpAction.value = 'bulk_delete';
                                form.appendChild(inpAction);
                                selected.forEach(cb => {
                                    const idInput = document.createElement('input');
                                    idInput.type = 'hidden'; idInput.name = 'ids[]'; idInput.value = cb.value;
                                    form.appendChild(idInput);
                                    const srcInput = document.createElement('input');
                                    srcInput.type = 'hidden'; srcInput.name = `sources[${cb.value}]`; srcInput.value = cb.getAttribute('data-source');
                                    form.appendChild(srcInput);
                                });
                                document.body.appendChild(form);
                                form.submit();
                            }
                            const bulkApprove = document.getElementById('bulkApprove');
                            if (bulkApprove) bulkApprove.addEventListener('click', () => submitBulk('approve'));
                            const bulkDelete = document.getElementById('bulkDelete');
                            if (bulkDelete) bulkDelete.addEventListener('click', submitBulkDelete);
                        })();
                    </script>
                    <div class="pagination">
                        <?php for ($i = 1; $i <= ceil($total_pendentes / $limit); $i++): ?>
                            <a href="<?php echo buildPaginationUrl($i); ?>" class="<?php echo $page == $i ? 'active' : ''; ?>">
                                <?php echo $i; ?>
                            </a>
                        <?php endfor; ?>
                    </div>
                <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-check-circle"></i>
                        <p>Não há comentários pendentes</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </main>


    <footer class="footer">
        <div class="container">
            <p class="text-center text-muted">© <?php echo date('Y'); ?> Sistema de Comentários</p>
        </div>
    </footer>

    <?php $conn->close(); ?>
    <script>
    // Copiar link ao clicar no campo de link, sem popup
    document.addEventListener('DOMContentLoaded', () => {
        const linkCells = document.querySelectorAll('td.col-link');
        linkCells.forEach(cell => {
            cell.addEventListener('click', () => {
                const link = cell.getAttribute('title') || cell.textContent.trim();
                if (!link) return;
                navigator.clipboard.writeText(link).catch(() => {});
            });
        });
    });
    </script>
    <script>
    // IA assíncrona: envia comentários em lote para backend OpenAI e atualiza badges com detalhe
    document.addEventListener('DOMContentLoaded', () => {
        // Prevenir execução múltipla
        if (window.iaAnalysisRunning) {
            console.log('Análise IA já está em execução');
            return;
        }
        window.iaAnalysisRunning = true;
        
        const pendingBadges = Array.from(document.querySelectorAll('.ai-badge.ai-pending'));
        if (pendingBadges.length === 0) {
            window.iaAnalysisRunning = false;
            return;
        }

        const badgeItems = pendingBadges
            .map(el => ({ el, comment: el.getAttribute('data-comment') || '', analysis: null }))
            .filter(({ comment }) => comment.trim() !== '');

        if (badgeItems.length === 0) {
            console.warn('Nenhum comentário válido para analisar');
            pendingBadges.forEach(el => {
                el.textContent = 'Sem texto';
                el.classList.remove('risk-neutral');
                el.classList.add('risk-blue');
            });
            window.iaAnalysisRunning = false;
            return;
        }

        const modal = document.getElementById('aiResultModal');
        const modalBody = document.getElementById('aiResultModalBody');
        const modalClose = document.getElementById('aiResultModalClose');

        const closeModal = () => {
            if (modal) {
                modal.classList.remove('open');
                modal.setAttribute('aria-hidden', 'true');
            }
        };

        const openModal = (detailData) => {
            if (!modal || !modalBody) {
                return;
            }
            modalBody.innerHTML = createDetailMarkup(detailData);
            modal.classList.add('open');
            modal.setAttribute('aria-hidden', 'false');
        };

        if (modalClose) {
            modalClose.addEventListener('click', closeModal);
        }

        if (modal) {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    closeModal();
                }
            });
        }

        badgeItems.forEach(item => {
            item.el.classList.add('clickable');
            item.el.title = 'Clique para ver o resultado completo';
            item.el.addEventListener('click', () => {
                if (!item.analysis) {
                    console.info('Resultado ainda está sendo gerado...');
                    return;
                }
                const detailData = item.analysis.raw || item.analysis;
                openModal(detailData);
            });
        });

        // Mostrar que está processando todos com indicador de progresso
        pendingBadges.forEach(el => {
            el.textContent = 'Aguarde...';
            el.classList.add('risk-neutral');
        });

        let secondsElapsed = 0;
        const startTime = Date.now();
        let timeInterval = null;
        let processedCount = 0;
        const totalToProcess = badgeItems.length;

        // Função para atualizar apenas badges pendentes
        const updatePendingBadges = () => {
            const stillPending = badgeItems.filter(item => !item.analysis);
            if (stillPending.length === 0) {
                if (timeInterval) {
                    clearInterval(timeInterval);
                    timeInterval = null;
                }
                return;
            }
            secondsElapsed = Math.floor((Date.now() - startTime) / 1000);
            stillPending.forEach(item => {
                item.el.textContent = `Analisando... ${secondsElapsed}s`;
            });
        };

        // Iniciar contador apenas para badges pendentes
        timeInterval = setInterval(updatePendingBadges, 1000);

        const comments = badgeItems.map(item => item.comment);
        console.log('Enviando', comments.length, 'comentários em lotes para análise');

        const BATCH_SIZE = 10;
        const batchResults = [];

        const applyResults = (results, offset) => {
            results.forEach((res, idx) => {
                const target = badgeItems[offset + idx];
                if (!target) return;
                const { el } = target;
                target.analysis = res;
                processedCount++;
                el.classList.remove('risk-neutral', 'risk-green', 'risk-yellow', 'risk-orange', 'risk-red', 'risk-blue');
                el.classList.add(res.color || 'risk-neutral');
                el.textContent = res.label || 'OK';
                const titleText = res.detail ? `${res.detail} — Clique para ver detalhes` : 'Clique para ver o resultado completo';
                el.title = titleText;
                const dot = el.parentElement && el.parentElement.querySelector('.ai-dot');
                if (dot) {
                    dot.classList.remove('risk-neutral', 'risk-green', 'risk-yellow', 'risk-orange', 'risk-red', 'risk-blue');
                    dot.classList.add(res.color || 'risk-neutral');
                }
            });
            
            // Parar contador se todos foram processados
            if (processedCount >= totalToProcess && timeInterval) {
                clearInterval(timeInterval);
                timeInterval = null;
            }
        };

        const sendBatch = async (batch, offset) => {
            const payload = { comments: batch };
            console.log(`Analisando lote ${offset / BATCH_SIZE + 1}: ${batch.length} comentários`);
            const response = await fetch('ia_analyze_openai.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload),
            });
            if (!response.ok) {
                const text = await response.text();
                console.log('Resposta de erro (text):', text);
                try {
                    const err = JSON.parse(text);
                    console.error('Erro da API:', err);
                    throw new Error(err.error || `HTTP ${response.status}`);
                } catch (e) {
                    throw new Error(`HTTP ${response.status} - ${text.substring(0, 100)}`);
                }
            }
            const data = await response.json();
            return data;
        };

        (async () => {
            try {
                for (let offset = 0; offset < comments.length; offset += BATCH_SIZE) {
                    const batch = comments.slice(offset, offset + BATCH_SIZE);
                    const data = await sendBatch(batch, offset);
                    const results = (data && data.results) ? data.results : [];
                    batchResults.push(...results);
                    applyResults(results, offset);
                }
                // Garantir que o contador seja parado
                if (timeInterval) {
                    clearInterval(timeInterval);
                    timeInterval = null;
                }
                console.log('Todos os lotes processados');
            } catch (error) {
                // Garantir que o contador seja parado mesmo em caso de erro
                if (timeInterval) {
                    clearInterval(timeInterval);
                    timeInterval = null;
                }
                console.error('Erro na análise IA:', error);
                console.error('Detalhes:', error.message || error);
                pendingBadges.forEach(el => {
                    el.textContent = 'Erro API';
                    el.classList.remove('risk-neutral', 'risk-green', 'risk-yellow', 'risk-orange', 'risk-red', 'risk-blue');
                    el.classList.add('risk-neutral');
                    el.title = error.message || 'Erro na análise';
                });
            } finally {
                // Resetar flag quando terminar
                window.iaAnalysisRunning = false;
            }
        })();
    });

    function createDetailMarkup(detailData) {
        const flagged = detailData?.flagged;
        const categories = detailData?.categories || {};
        const scores = detailData?.category_scores || {};
        const detectedCategories = Object.entries(categories).filter(([, value]) => value);
        const allCategories = Object.entries(categories);

        let html = `
            <div class="ai-detail">
                <div class="ai-detail-header">
                    <h3>${flagged ? 'Comentário sinalizado' : 'Comentário aprovado'}</h3>
                    <span class="badge bg-secondary">ID: ${detailData?.comment_id || '&#x2014;'}</span>
                </div>
                <div class="ai-detail-comment">${escapeHtml(detailData?.content || detailData?.comment || '')}</div>
                <div>
                    <strong>Categorias detectadas:</strong>
                    <div class="ai-detail-badges">
        `;

        if (detectedCategories.length === 0) {
            html += `<span class="ai-category-badge not-detected">Nenhuma categoria crítica detectada</span>`;
        } else {
            detectedCategories.forEach(([category]) => {
                const score = scores[category] || 0;
                html += `
                    <span class="ai-category-badge detected" title="Pontuação: ${formatScore(score)}">
                        ${formatCategory(category)} • ${formatScore(score)}
                    </span>
                `;
            });
        }

        html += `
                    </div>
                </div>
                <div class="mt-3">
                    <strong>Todas as categorias:</strong>
                    <div class="ai-detail-badges">
        `;

        if (allCategories.length === 0) {
            html += `<span class="ai-category-badge not-detected">Nenhuma categoria cadastrada</span>`;
        } else {
            allCategories.forEach(([category]) => {
                const score = scores[category] || 0;
                const badgeClass = categories[category] ? 'detected' : 'not-detected';
                html += `
                    <span class="ai-category-badge ${badgeClass}">
                        ${formatCategory(category)} • ${formatScore(score)}
                    </span>
                `;
            });
        }

        html += `
                    </div>
                </div>
            </div>
        `;

        return html;
    }

    function formatCategory(category) {
        const translations = {
            'hate': 'Ódio',
            'hate/threatening': 'Ódio/Ameaçador',
            'harassment': 'Assédio',
            'harassment/threatening': 'Assédio/Ameaçador',
            'self-harm': 'Autolesão',
            'self-harm/intent': 'Autolesão/Intenção',
            'self-harm/instructions': 'Autolesão/Instruções',
            'sexual': 'Sexual',
            'sexual/minors': 'Sexual/Menores',
            'violence': 'Violência',
            'violence/graphic': 'Violência/Gráfico',
            'illicit': 'Ilícito',
            'illicit/violent': 'Ilícito/Violento'
        };
        return translations[category] || category;
    }

    function formatScore(score) {
        if (typeof score !== 'number') {
            score = parseFloat(score) || 0;
        }
        return `${(score * 100).toFixed(2)}%`;
    }

    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    </script>
    <div id="aiResultModal" class="ai-modal" aria-hidden="true">
        <div class="ai-modal-content">
            <button id="aiResultModalClose" class="ai-modal-close" aria-label="Fechar resultado">×</button>
            <div id="aiResultModalBody"></div>
        </div>
    </div>
</body>
</html> 
</html> 